package com.aliyun.dashvector.models;

import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.common.ErrorCode;
import com.aliyun.dashvector.proto.CollectionInfo;
import com.aliyun.dashvector.proto.FieldType;
import com.aliyun.dashvector.proto.Status;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import com.aliyun.dashvector.utils.Convertor;
import com.aliyun.dashvector.utils.Utils;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.Getter;
import org.apache.commons.collections.MapUtils;

import static com.aliyun.dashvector.utils.Validator.RESERVE_DENSE_VECTOR;

/**
 * @author sanyi
 */
@Getter
public class CollectionMeta implements Serializable {
  private final String name;
  @JsonProperty("dimension")
  private final int dimension;
  @JsonProperty("data_type")
  private final CollectionInfo.DataType dataType;
  @JsonProperty("metric")
  private final CollectionInfo.Metric metric;
  private final String status;
  @JsonProperty("fields_schema")
  private final Map<String, FieldType> fieldsSchema;
  @JsonProperty("partition_status")
  private final Map<String, Status> partitionStatus;
  @JsonProperty("vectors")
  private final Map<String, VectorParam> vectors;
  @JsonProperty("sparse_vectors")
  private final Map<String, VectorParam> sparseVectors;

  @Override
  public String toString() {
    return Utils.toString(this);
  }

  @JsonIgnore
  public int getDimension() {
    checkMultiVectors();
    return dimension;
  }

  public int getDimension(String vectorName) {
    return getVectorParam(vectorName).getDimension();
  }

  @JsonIgnore
  public CollectionInfo.Metric getMetric() {
    checkMultiVectors();
    return metric;
  }

  @JsonIgnore
  public CollectionInfo.Metric getMetric(String sparseVectorName) {
    return getVectorParam(sparseVectorName).getMetric();
  }

  @JsonIgnore
  public CollectionInfo.DataType getDataType() {
    checkMultiVectors();
    return dataType;
  }

  public CollectionInfo.DataType getDataType(String vectorName) {
    return getVectorParam(vectorName).getDataType();
  }

  public CollectionMeta(CollectionInfo collectionInfo) {
    this.name = collectionInfo.getName();
    this.status = collectionInfo.getStatus().name();
    this.fieldsSchema = collectionInfo.getFieldsSchemaMap();
    this.partitionStatus = collectionInfo.getPartitionsMap();

    int dimension = collectionInfo.getDimension();
    CollectionInfo.DataType dataType = collectionInfo.getDtype();
    CollectionInfo.Metric metric = collectionInfo.getMetric();
    Map<String, VectorParam> vectorsMap = new HashMap<>();

    CollectionInfo.VectorParam denseVectorParam = collectionInfo.getVectorsSchemaMap().get(RESERVE_DENSE_VECTOR);

    if (denseVectorParam != null) {
      dimension = denseVectorParam.getDimension();
      dataType = denseVectorParam.getDtype();
      metric = denseVectorParam.getMetric();
    } else if (MapUtils.isNotEmpty(collectionInfo.getVectorsSchemaMap())) {
      vectorsMap = collectionInfo.getVectorsSchemaMap().entrySet().stream()
              .collect(Collectors.toMap(
                      Map.Entry::getKey,
                      entry -> Convertor.toVectorParam(entry.getValue())
              ));
    }

    this.dimension = dimension;
    this.dataType = dataType;
    this.metric = metric;
    this.vectors = vectorsMap;
    this.sparseVectors = collectionInfo.getSparseVectorsSchemaMap().entrySet().stream()
            .collect(Collectors.toMap(
                    Map.Entry::getKey,
                    entry -> Convertor.toVectorParam(entry.getValue())
            ));
  }

  private void checkMultiVectors() {
    if (MapUtils.isNotEmpty(vectors)) {
      if (vectors.size() > 1) {
        throw new DashVectorException(ErrorCode.INVALID_ARGUMENT.getCode(),
                "DashVectorSDK should provide vector name when collection has multi dense vectors");
      }
    }
    if (MapUtils.isNotEmpty(sparseVectors)) {
      throw new DashVectorException(ErrorCode.INVALID_ARGUMENT.getCode(),
              "DashVectorSDK should provide vector name when collection has multi sparse vectors");
    }
  }

  public VectorParam getVectorParam(String vectorName) {
    if (vectors.containsKey(vectorName)) {
      return vectors.get(vectorName);
    }
    if (sparseVectors.containsKey(vectorName)) {
      return sparseVectors.get(vectorName);
    }
    throw new DashVectorException(ErrorCode.INVALID_ARGUMENT.getCode(),
            String.format("DashVectorSDK vector name %s not found", vectorName));
  }
}
