package com.aliyun.dashvector.models.responses;

import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.common.ErrorCode;
import com.aliyun.dashvector.models.*;
import com.aliyun.dashvector.models.Doc;
import com.aliyun.dashvector.models.DocOpResult;
import com.aliyun.dashvector.models.PartitionStats;
import com.aliyun.dashvector.models.RequestUsage;
import com.aliyun.dashvector.proto.*;
import com.aliyun.dashvector.utils.Convertor;
import com.aliyun.dashvector.utils.Utils;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.grpc.StatusRuntimeException;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import io.grpc.Metadata;
import lombok.Getter;
import org.apache.commons.lang3.StringUtils;

/**
 * @author sanyi
 */
@Getter
public class Response<T> implements Serializable {
  private final int code;
  private final String message;
  @JsonProperty("request_id")
  private final String requestId;

  @JsonInclude(JsonInclude.Include.NON_NULL)
  private final T output;

  @JsonInclude(JsonInclude.Include.NON_NULL)
  private final RequestUsage usage;

  private static final Metadata.Key<String> EXTRA_CODE_KEY =
          Metadata.Key.of("extra-code", Metadata.ASCII_STRING_MARSHALLER);
  private static final Metadata.Key<String> REQUEST_ID_KEY =
          Metadata.Key.of("request-id", Metadata.ASCII_STRING_MARSHALLER);

  @Override
  public String toString() {
    return Utils.toString(this);
  }

  private Response(int code, String message, String requestId, T output) {
    this(code, message, requestId, output, null);
  }

  private Response(int code, String message, String requestId, T output, RequestUsage usage) {
    this.code = code;
    this.message = message;
    this.requestId = requestId;
    this.output = output;
    this.usage = usage;
  }

  @JsonIgnore
  public Boolean isSuccess() {
    return this.code == ErrorCode.SUCCESS.getCode();
  }

  public static <T> Response<T> create(int code, String message, String requestId, T output) {
    return new Response<>(code, message, requestId, output);
  }

  public static <T> Response<T> create(int code, String message, String requestId, T output, RequestUsage usage) {
    return new Response<>(code, message, requestId, output, usage);
  }

  public static <T> Response<T> failed(Exception e) {
    if (e instanceof DashVectorException) {
      return handleDashVectorException((DashVectorException) e);
    } else if (e instanceof ExecutionException) {
      return handleExecutionException((ExecutionException) e);
    }
    return handleOtherException(e);
  }

  private static <T> Response<T> handleDashVectorException(DashVectorException e) {
    return create(e.getCode(), e.getMessage(), null, null);
  }

  private static <T> Response<T> handleExecutionException(ExecutionException e) {
    Throwable cause = e.getCause();
    if (cause instanceof StatusRuntimeException) {
      return handleStatusRuntimeException((StatusRuntimeException) cause);
    }
    return handleOtherException(e);
  }

  private static <T> Response<T> handleStatusRuntimeException(StatusRuntimeException e) {
    Metadata trailers = e.getTrailers();
    // user grpc StatusRuntimeException error code
    //   OK(0),
    //   CANCELLED(1),
    //   UNKNOWN(2),
    //   INVALID_ARGUMENT(3),
    //   DEADLINE_EXCEEDED(4),
    //   NOT_FOUND(5),
    //   ALREADY_EXISTS(6),
    //   PERMISSION_DENIED(7),
    //   RESOURCE_EXHAUSTED(8),
    //   FAILED_PRECONDITION(9),
    //   ABORTED(10),
    //   OUT_OF_RANGE(11),
    //   UNIMPLEMENTED(12),
    //   INTERNAL(13),
    //   UNAVAILABLE(14),
    //   DATA_LOSS(15),
    //   UNAUTHENTICATED(16)
    int code =
        Optional.ofNullable(trailers)
            .map(trailer -> trailer.get(EXTRA_CODE_KEY))
            .map(Integer::parseInt)
            .orElse(e.getStatus().getCode().value());
    String requestId = Optional.ofNullable(trailers)
            .map(trailer -> trailer.get(REQUEST_ID_KEY))
            .orElse(StringUtils.EMPTY);
    return create(code, e.getStatus().getDescription(), requestId, null);
  }

  private static <T> Response<T> handleOtherException(Exception e) {
    return create(ErrorCode.UNKNOWN.getCode(), e.getMessage(), null, null);
  }

  public static Response<Void> success(int code, String message, String requestId) {
    return create(code, message, requestId, null);
  }

  public static Response<List<String>> success(ListCollectionsResponse response) {
    return create(
        response.getCode(),
        response.getMessage(),
        response.getRequestId(),
        response.getOutputList());
  }

  public static Response<CollectionMeta> success(DescribeCollectionResponse response) {
    return create(
        response.getCode(),
        response.getMessage(),
        response.getRequestId(),
        new CollectionMeta(response.getOutput()));
  }

  public static Response<List<DocOpResult>> success(InsertDocResponse response) {
    List<DocOpResult> docOpResults = response.getOutputList().stream().map(DocOpResult::new).collect(Collectors.toList());
    return response.hasUsage() ?
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults, Convertor.toRequestUsage(response.getUsage())) :
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults);
  }

  public static Response<List<DocOpResult>> success(UpsertDocResponse response) {
    List<DocOpResult> docOpResults = response.getOutputList().stream().map(DocOpResult::new).collect(Collectors.toList());
    return response.hasUsage() ?
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults, Convertor.toRequestUsage(response.getUsage())) :
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults);
  }

  public static Response<List<DocOpResult>> success(UpdateDocResponse response) {
    List<DocOpResult> docOpResults = response.getOutputList().stream().map(DocOpResult::new).collect(Collectors.toList());
    return response.hasUsage() ?
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults, Convertor.toRequestUsage(response.getUsage())) :
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults);
  }

  public static Response<List<DocOpResult>> success(DeleteDocResponse response) {
    List<DocOpResult> docOpResults = response.getOutputList().stream().map(DocOpResult::new).collect(Collectors.toList());
    return response.hasUsage() ?
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults, Convertor.toRequestUsage(response.getUsage())) :
        create(response.getCode(), response.getMessage(), response.getRequestId(), docOpResults);
  }

  public static Response<List<Doc>> success(
          QueryDocResponse response, CollectionMeta collectionMeta) {
    List<Doc> docs =
        response.getOutputList().stream()
            .map(doc -> Convertor.fromDoc(doc, collectionMeta))
            .collect(Collectors.toList());
    return response.hasUsage() ?
        create(response.getCode(), response.getMessage(), response.getRequestId(), docs, Convertor.toRequestUsage(response.getUsage())) :
        create(response.getCode(), response.getMessage(), response.getRequestId(), docs);
  }

  public static Response<List<Group>> success(
          QueryDocGroupByResponse response, CollectionMeta collectionMeta) {
    List<Group> groups =
            response.getOutputList().stream()
                    .map(group -> Convertor.fromGroup(group, collectionMeta))
                    .collect(Collectors.toList());
    return create(response.getCode(), response.getMessage(), response.getRequestId(), groups);
  }

  public static Response<Map<String, Doc>> success(
          FetchDocResponse response, CollectionMeta collectionMeta) {
    Map<String, Doc> docMap =
        response.getOutputMap().values().stream()
            .map(doc -> Convertor.fromDoc(doc, collectionMeta))
            .collect(Collectors.toMap(Doc::getId, v -> v));
    return response.hasUsage() ?
        create(response.getCode(), response.getMessage(), response.getRequestId(), docMap, Convertor.toRequestUsage(response.getUsage())) :
        create(response.getCode(), response.getMessage(), response.getRequestId(), docMap);
  }

  public static Response<CollectionStats> success(StatsCollectionResponse response) {
    return create(
        response.getCode(),
        response.getMessage(),
        response.getRequestId(),
        new CollectionStats(response.getOutput()));
  }

  public static Response<Status> success(DescribePartitionResponse response) {
    return create(
        response.getCode(), response.getMessage(), response.getRequestId(), response.getOutput());
  }

  public static Response<List<String>> success(ListPartitionsResponse response) {
    return create(
        response.getCode(),
        response.getMessage(),
        response.getRequestId(),
        response.getOutputList());
  }

  public static Response<PartitionStats> success(StatsPartitionResponse response) {
    return create(
        response.getCode(),
        response.getMessage(),
        response.getRequestId(),
        new PartitionStats(response.getOutput()));
  }

  public static Response<Void> success(DeletePartitionResponse response) {
    return new Response<>(response.getCode(), response.getMessage(), response.getRequestId(), null);
  }
}
