package com.aliyun.dashvector.validator;

import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.common.ErrorCode;
import com.aliyun.dashvector.models.requests.CreateCollectionRequest;
import com.aliyun.dashvector.proto.CollectionInfo;
import com.aliyun.dashvector.utils.Validator;

import static com.aliyun.dashvector.utils.Validator.*;

/**
 * @author sanyi
 * @date 2025/5/28 17:42
 */
public abstract class CreateCollectionValidator {

    protected final CreateCollectionRequest request;

    protected CreateCollectionValidator(CreateCollectionRequest request) {
        this.request = request;
    }

    protected abstract void validateVectorSchema();

    public void validate() {
        // validate collection name
        verifyCollectionName(request.getName());
        // validate vector schema
        validateVectorSchema();
        // validate fields
        verifyFields(request.getFiledsSchema());
    }

    public static class ReserveCreateCollectionValidator extends CreateCollectionValidator {

        protected ReserveCreateCollectionValidator(CreateCollectionRequest request) {
            super(request);
        }

        @Override
        protected void validateVectorSchema() {
            if (request.getVectorParam() != null) {
                if (request.getDimension() > 0) {
                    throw new DashVectorException(ErrorCode.INVALID_ARGUMENT,
                            "DashVectorSDK does not support passing in dimension and vector_param simultaneously");
                }
                verifyVectorParam(request.getVectorParam());
            } else {
                Validator.verifyDimension(request.getDimension());
                if (request.getMetric() == CollectionInfo.Metric.cosine
                        && request.getDataType() != CollectionInfo.DataType.FLOAT) {
                    throw new DashVectorException(
                            ErrorCode.MISMATCHED_DATA_TYPE.getCode(),
                            "DashVectorSDK CreateCollectionRequest datatype is Invalid and must be [float] when metric is cosine");
                }
            }
        }
    }

    public static class MultiVectorCreateCollectionValidator extends CreateCollectionValidator {

        protected MultiVectorCreateCollectionValidator(CreateCollectionRequest request) {
            super(request);
        }

        @Override
        protected void validateVectorSchema() {
            verifyVectorNameAndCount(request.getVectors());
        }
    }
}
