package com.aliyun.dashvector.validator;

import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.common.ErrorCode;
import com.aliyun.dashvector.models.*;
import com.aliyun.dashvector.models.requests.QueryDocGroupByRequest;
import com.aliyun.dashvector.models.requests.QueryDocRequest;
import com.aliyun.dashvector.proto.CollectionInfo;
import com.aliyun.dashvector.proto.DocOpResult;
import org.apache.commons.collections.MapUtils;

import java.util.Map;
import java.util.Objects;

import static com.aliyun.dashvector.utils.Validator.*;

/**
 * @author sanyi
 * @date 2025/5/28 15:06
 */
public class MultiVectorValidator extends VectorValidator{

    protected MultiVectorValidator(CollectionMeta meta) {
        super(meta);
    }

    @Override
    protected void validateVectorFromDoc(Doc doc, DocOpResult.DocOp docOp) {
        Map<String, Vector> vectors = doc.getVectors();
        Map<String, SparseVector> sparseVectors = doc.getSparseVectors();
        if (MapUtils.isEmpty(vectors) && MapUtils.isEmpty(sparseVectors)) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Doc without vector");
        }

        // validate vector size
        verifyVectorCount(vectors.size() + sparseVectors.size());

        // validate dense vector
        if (doc.getVector() != null) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Collection has multi vector but provide " +
                    "single vector in doc");
        }
        if (vectors.containsKey(RESERVE_DENSE_VECTOR)) {
            throw new DashVectorException(ErrorCode.INVALID_VECTOR_FORMAT, RESERVE_DENSE_VECTOR_MSG);
        }
        vectors.forEach((k, v) -> {
            VectorParam vectorParam = meta.getVectors().get(k);
            if (vectorParam == null) {
                throw new DashVectorException(
                        ErrorCode.INVALID_ARGUMENT,
                        "Vector name " + k + " is not exist");
            }
            verifyVector(v, docOp, vectorParam.getDimension());
        });

        // validate sparse vector
        if (MapUtils.isNotEmpty(doc.getSparseVector())) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT,
                    "Collection has multi vector but provide single sparse vector in doc");
        }
        if  (sparseVectors.containsKey(RESERVE_SPARSE_VECTOR)) {
            throw new DashVectorException(ErrorCode.INVALID_VECTOR_FORMAT, RESERVE_SPARSE_VECTOR_MSG);
        }

        sparseVectors.forEach((k, v) -> {
            if (!meta.getSparseVectors().containsKey(k)) {
                throw new DashVectorException(
                        ErrorCode.INVALID_ARGUMENT,
                        "Sparse vector name " + k + " is not exist");
            }
            verifySparseVector(k, v.getValue(), meta);
        });
    }

    @Override
    protected void validateVectorFromQuery(QueryDocRequest request) {
        Map<String, VectorQuery> vectors = request.getVectors();
        Map<String, SparseVectorQuery> sparseVectors = request.getSparseVectors();
        if (MapUtils.isEmpty(vectors) && MapUtils.isEmpty(sparseVectors)) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Query request without vector");
        }

        // validate vector size
        verifyVectorCount(vectors.size() + sparseVectors.size());

        // validate dense vector
        if (request.getVector() != null) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Collection has multi vector and use vectors " +
                    "to query");
        }
        if (vectors.containsKey(RESERVE_DENSE_VECTOR)) {
            throw new DashVectorException(ErrorCode.INVALID_VECTOR_FORMAT, RESERVE_DENSE_VECTOR_MSG);
        }
        request.getVectors().forEach((k, v) -> {
            VectorParam vectorParam = meta.getVectors().get(k);
            if (vectorParam == null) {
                throw new DashVectorException(
                        ErrorCode.INVALID_ARGUMENT,
                        "Vector name " + k + " is not exist");
            }
            verifyVector(v.getVector(), DocOpResult.DocOp.UNRECOGNIZED, vectorParam.getDimension());
        });

        // validate sparse vector
        if (request.getSparseVector() != null) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Collection has multi vector and use " +
                    "sparse_vector to query");
        }
        if (vectors.containsKey(RESERVE_SPARSE_VECTOR)) {
            throw new DashVectorException(ErrorCode.INVALID_VECTOR_FORMAT, RESERVE_SPARSE_VECTOR_MSG);
        }
        request.getSparseVectors().forEach((k, v) -> {
            if (!meta.getSparseVectors().containsKey(k)) {
                throw new DashVectorException(
                        ErrorCode.INVALID_ARGUMENT,
                        "Sparse vector name " + k + " is not exist");
            }
            verifySparseVector(k, v.getVector().getValue(), meta);
        });
    }

    @Override
    protected void validateVectorFromQueryGroupBy(QueryDocGroupByRequest request) {
        // validate dense vector
        if (request.getVector() != null) {
            if (Objects.equals(request.getVectorField(), RESERVE_DENSE_VECTOR)) {
                throw new DashVectorException(
                        ErrorCode.INVALID_GROUP_BY,
                        "Collection does not support query group by reverse vector field"
                );
            }
            VectorParam vectorParam = meta.getVectors().get(request.getVectorField());
            if (vectorParam == null) {
                throw new DashVectorException(
                        ErrorCode.INVALID_GROUP_BY,
                        "Vector name " + request.getVectorField() + " is not exist");
            }
            verifyVector(request.getVector(), DocOpResult.DocOp.UNRECOGNIZED, vectorParam.getDimension());
        }

        // validate sparse vector
        if (request.getSparseVector() != null) {
            if (Objects.equals(request.getVectorField(), RESERVE_SPARSE_VECTOR)) {
                throw new DashVectorException(
                        ErrorCode.INVALID_GROUP_BY,
                        "Collection does not support query group by reverse vector field"
                );
            }
            if (!meta.getSparseVectors().containsKey(request.getVectorField())) {
                throw new DashVectorException(
                        ErrorCode.INVALID_GROUP_BY,
                        "Sparse vector name " + request.getVectorField() + " is not exist");
            }
            verifySparseVector(request.getSparseVector(), meta);
        }
    }
}
