package com.aliyun.dashvector.validator;

import com.aliyun.dashvector.common.DashVectorException;
import com.aliyun.dashvector.common.ErrorCode;
import com.aliyun.dashvector.models.CollectionMeta;
import com.aliyun.dashvector.models.Doc;
import com.aliyun.dashvector.models.PartitionStats;
import com.aliyun.dashvector.models.requests.QueryDocGroupByRequest;
import com.aliyun.dashvector.models.requests.QueryDocRequest;
import com.aliyun.dashvector.proto.DocOpResult;
import com.aliyun.dashvector.utils.Validator;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

import static com.aliyun.dashvector.utils.Validator.*;

/**
 * @author sanyi
 * @date 2025/5/28 14:37
 */
public class ReserveVectorValidator extends VectorValidator{

    protected ReserveVectorValidator(CollectionMeta meta) {
        super(meta);
    }

    @Override
    protected void validateVectorFromDoc(Doc doc, DocOpResult.DocOp docOp) {
        // validate dense vector
        if (MapUtils.isNotEmpty(doc.getVectors())) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Collection has only single vector but provide " +
                    "multi vector in doc");
        } else {
            verifyVector(doc.getVector(), docOp, meta.getDimension());
        }

        // validate sparse vector
        verifySparseVector(doc.getSparseVector(), meta);
    }

    @Override
    protected void validateVectorFromQuery(QueryDocRequest request) {
        // validate dense vector
        if (MapUtils.isNotEmpty(request.getVectors())) {
            throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Collection has only single vector but provide " +
                    "multi vector in query");
        }
        if (request.getVectorQuery() != null) {
            if (request.getVector() != null) {
                throw new DashVectorException(ErrorCode.INVALID_ARGUMENT, "Collection has only single vector but provide " +
                        "vector and vector_query");
            }
            // query by vector_query
            verifyVector(request.getVectorQuery().getVector(), DocOpResult.DocOp.UNRECOGNIZED, meta.getDimension());
            verifyTopK(request.getVectorQuery().getNumCandidates(), request.isIncludeVector());
        } else {
            // query by vector
            verifyVector(request.getVector(), DocOpResult.DocOp.UNRECOGNIZED, meta.getDimension());
            verifyTopK(request.getTopk(), request.isIncludeVector());
        }

        // validate sparse vector
        verifySparseVector(request.getSparseVector(), meta);
    }

    @Override
    protected void validateVectorFromQueryGroupBy(QueryDocGroupByRequest request) {
        if (StringUtils.isNotEmpty(request.getVectorField())) {
            throw new DashVectorException(
                    ErrorCode.INVALID_GROUP_BY,
                    "Collection has only vector but does not support query group by vector field");
        }

        // validate dense vector
        verifyVector(request.getVector(), DocOpResult.DocOp.UNRECOGNIZED, meta.getDimension());

        // validate sparse vector
        verifySparseVector(request.getSparseVector(), meta);
    }
}
