/*
 * Decompiled with CFR 0.152.
 */
package com.apple.itunes.storekit.verification;

import com.apple.itunes.storekit.verification.AppleExtensionCertPathChecker;
import com.apple.itunes.storekit.verification.VerificationException;
import com.apple.itunes.storekit.verification.VerificationStatus;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.PublicKey;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXCertPathValidatorResult;
import java.security.cert.PKIXParameters;
import java.security.cert.PKIXRevocationChecker;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.time.Clock;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Base64;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

public class ChainVerifier {
    private static final int EXPECTED_CHAIN_LENGTH = 3;
    private static final int MAXIMUM_CACHE_SIZE = 32;
    private static final int CACHE_TIME_LIMIT = 15;
    private final Set<TrustAnchor> trustAnchors;
    private final ConcurrentHashMap<List<String>, CachedEntry> verifiedPublicKeyCache;
    private final Clock clock;

    public ChainVerifier(Set<InputStream> rootCertificates) {
        this(rootCertificates, Clock.systemUTC());
    }

    ChainVerifier(Set<InputStream> rootCertificates, Clock clock) {
        try {
            CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
            this.trustAnchors = new HashSet<TrustAnchor>();
            for (InputStream inputStream : rootCertificates) {
                Certificate certificate = certificateFactory.generateCertificate(inputStream);
                if (!(certificate instanceof X509Certificate)) {
                    throw new RuntimeException("Root certificate not of the expected X509 format");
                }
                this.trustAnchors.add(new TrustAnchor((X509Certificate)certificate, null));
            }
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        if (this.trustAnchors.size() == 0) {
            throw new RuntimeException("At least one root certificate is required");
        }
        this.verifiedPublicKeyCache = new ConcurrentHashMap();
        this.clock = clock;
    }

    public PublicKey verifyChain(String[] certificates, boolean performRevocationChecking, Date effectiveDate) throws VerificationException {
        PublicKey cachedKey;
        if (performRevocationChecking && certificates.length > 0 && (cachedKey = this.getCachedPrivateKey(Arrays.asList(certificates))) != null) {
            return cachedKey;
        }
        PublicKey publicKey = this.verifyChainWithoutCaching(certificates, performRevocationChecking, effectiveDate);
        if (performRevocationChecking) {
            this.putVerifiedPublicKey(Arrays.asList(certificates), publicKey);
        }
        return publicKey;
    }

    PublicKey verifyChainWithoutCaching(String[] certificates, boolean performRevocationChecking, Date effectiveDate) throws VerificationException {
        CertPathValidator certPathValidator;
        CertificateFactory certificateFactory;
        try {
            certificateFactory = CertificateFactory.getInstance("X.509");
            certPathValidator = CertPathValidator.getInstance("PKIX");
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        LinkedList<Certificate> parsedCertificates = new LinkedList<Certificate>();
        try {
            for (String c : certificates) {
                ByteArrayInputStream derEncodedCert = new ByteArrayInputStream(Base64.getDecoder().decode(c));
                parsedCertificates.add(certificateFactory.generateCertificate(derEncodedCert));
            }
        }
        catch (Exception e) {
            throw new VerificationException(VerificationStatus.INVALID_CERTIFICATE, (Throwable)e);
        }
        if (parsedCertificates.size() != 3) {
            throw new VerificationException(VerificationStatus.INVALID_CHAIN_LENGTH);
        }
        try {
            CertPath certPath = certificateFactory.generateCertPath(parsedCertificates.subList(0, 2));
            PKIXParameters parameters = new PKIXParameters(this.trustAnchors);
            parameters.setRevocationEnabled(performRevocationChecking);
            parameters.setDate(effectiveDate);
            if (performRevocationChecking) {
                PKIXRevocationChecker revocationChecker = (PKIXRevocationChecker)certPathValidator.getRevocationChecker();
                revocationChecker.setOptions(Set.of(PKIXRevocationChecker.Option.NO_FALLBACK));
                parameters.addCertPathChecker(revocationChecker);
            }
            parameters.addCertPathChecker(new AppleExtensionCertPathChecker());
            PKIXCertPathValidatorResult certPathValidatorResult = (PKIXCertPathValidatorResult)certPathValidator.validate(certPath, parameters);
            return certPathValidatorResult.getPublicKey();
        }
        catch (Exception e) {
            throw new VerificationException(VerificationStatus.INVALID_CHAIN, (Throwable)e);
        }
    }

    private PublicKey getCachedPrivateKey(List<String> certificateChain) {
        if (this.verifiedPublicKeyCache.containsKey(certificateChain) && this.verifiedPublicKeyCache.get(certificateChain).cachedExpirationDate.isAfter(this.clock.instant())) {
            return this.verifiedPublicKeyCache.get(certificateChain).publicKey;
        }
        return null;
    }

    private void putVerifiedPublicKey(List<String> certificateChain, PublicKey publicKey) {
        Instant cacheExpiration = this.clock.instant().plus(15L, ChronoUnit.MINUTES);
        this.verifiedPublicKeyCache.put(certificateChain, new CachedEntry(cacheExpiration, publicKey));
        if (this.verifiedPublicKeyCache.size() > 32) {
            this.verifiedPublicKeyCache.entrySet().removeIf(e -> ((CachedEntry)e.getValue()).cachedExpirationDate.isBefore(this.clock.instant()));
        }
    }

    private static class CachedEntry {
        private final Instant cachedExpirationDate;
        private final PublicKey publicKey;

        public CachedEntry(Instant cachedExpirationDate, PublicKey publicKey) {
            this.cachedExpirationDate = cachedExpirationDate;
            this.publicKey = publicKey;
        }
    }
}

