package com.bxm.report.web.controller;

import java.util.ArrayList;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.app.model.dto.PositionClassTypeDto;
import com.bxm.app.model.dto.QueryAppAdPositionDto;
import com.bxm.app.model.ro.AppEntranceAdRo;
import com.bxm.app.model.ro.PositionClassType;
import com.bxm.datapark.facade.Page;
import com.bxm.datapark.facade.position.model.dto.PositionClassCountDto;
import com.bxm.datapark.facade.position.model.dto.PositionHourCountDto;
import com.bxm.datapark.facade.position.model.dto.PositionSpareTicketDto;
import com.bxm.datapark.facade.position.model.dto.PositionUserPortraitDto;
import com.bxm.datapark.facade.position.model.vo.PositionClassCountVo;
import com.bxm.datapark.facade.position.model.vo.PositionHourCountVo;
import com.bxm.datapark.facade.position.model.vo.PositionSpareTicketVo;
import com.bxm.datapark.facade.position.model.vo.PositionUserPortraitVo;
import com.bxm.report.facade.app.AppEntranceFacadeIntegration;
import com.bxm.report.facade.app.MediaCommonConfigFacadeIntegration;
import com.bxm.report.facade.datapark.DataparkPositionPullerIntegration;
import com.bxm.report.model.dao.monitor.WarnConfig;
import com.bxm.report.model.enums.WarnConfigEnum;
import com.bxm.report.service.monitor.PositionWarnConfigService;
import com.bxm.report.service.monitor.WarnConfigService;
import com.bxm.report.service.utils.ResultModelFactory;
import com.bxm.util.dto.ResultModel;

@RestController
@RequestMapping("/positionReport")
public class PositionReportController {

    private static final Logger logger = LoggerFactory.getLogger(PositionReportController.class);

    @Autowired
    private PositionWarnConfigService positionWarnConfigService;

    @Autowired
    private WarnConfigService warnConfigService;

    @Autowired
    private DataparkPositionPullerIntegration positionPullerIntegration;

    @Autowired
    private MediaCommonConfigFacadeIntegration mediaCommonConfigFacadeIntegration;

    @Autowired
    private AppEntranceFacadeIntegration appEntranceFacadeIntegration;
    
    /**
    * @Title: analysisBusinessOrder  
    * @Description: 广告位备用券统计报表
    * @param PositionSpareTicketDto
    * @return ResultModel<Page<PositionSpareTicketVo>> 
     */
    @RequestMapping(value = "/spareTicket", method = RequestMethod.GET)
    public ResultModel<Page<PositionSpareTicketVo>> analysisBusinessOrder(HttpServletRequest request,
            PositionSpareTicketDto param) {
        if (null == param) {
            param = new PositionSpareTicketDto();
        }
        try {
            //查询配置的阈值
            WarnConfig warnConfig = warnConfigService
                    .selectByConfigKey(WarnConfigEnum.POSITION_SPARE_TICKET.getConfigKey());
            if (warnConfig != null && StringUtils.isNotBlank(warnConfig.getPropValue())) {
                String propValue = warnConfig.getPropValue();
                String[] param1 = propValue.split(",");
                if (StringUtils.isNumeric(param1[0])) {
                    param.setOpenPv(Double.valueOf(param1[0]));
                }
                if (StringUtils.isNotBlank(param1[1])) {
                    String[] param2 = param1[1].split("-");
                    if (StringUtils.isNotBlank(param2[0]) && StringUtils.isNumeric(param2[0])) {
                        param.setSpareOpenProportionMin(Double.valueOf(param2[0]));
                    }
                    if (StringUtils.isNotBlank(param2[1]) && StringUtils.isNumeric(param2[1])) {
                        param.setSpareOpenProportionMax(Double.valueOf(param2[1]));
                    }
                }
            }
            //查询配置的排除广告位
            List<String> positionIds = positionWarnConfigService
                    .getAllConfigPositionId(WarnConfigEnum.POSITION_SPARE_TICKET.getConfigKey());
            if (!CollectionUtils.isEmpty(positionIds)) {
                StringBuffer sb = new StringBuffer();
                for (int i = 0; i < positionIds.size(); i++) {
                    if (i > 0) {
                        sb.append(",");
                    }
                    sb.append(positionIds.get(i));
                }
                param.setExcludePositionIds(sb.toString());
            }
        } catch (Exception e) {
            logger.error("查询广告位备用券报表数据失败" + e.getMessage(), e);
            return ResultModelFactory.FAIL();
        }
        return positionPullerIntegration.analysisPositionSpareTicket(param);
    }

    /**
    * @Title: analysisPositionUserPortrait  
    * @Description: 分页
    * @param PositionUserPortraitDto param
    * @return ResultModel<Page<PositionUserPortraitVo>> 
     */
    @RequestMapping(value = "/userPortrait", method = RequestMethod.GET)
    public ResultModel<Page<PositionUserPortraitVo>> analysisPositionUserPortrait(HttpServletRequest request,
            PositionUserPortraitDto param) {
        if (null == param) {
            param = new PositionUserPortraitDto();
        }
        return positionPullerIntegration.analysisPositionUserPortrait(param);
    }

    /**
    * @Title: analysisPositionUserPortraitNoPage  
    * @Description: 不分页
    * @param PositionUserPortraitDto param
    * @return ResultModel<List<PositionUserPortraitVo>> 
     */
    @RequestMapping(value = "/userPortraitNoPage", method = RequestMethod.GET)
    public ResultModel<List<PositionUserPortraitVo>> analysisPositionUserPortraitNoPage(HttpServletRequest request,
            PositionUserPortraitDto param) {
        if (null == param) {
            param = new PositionUserPortraitDto();
        }
        ResultModel<List<PositionUserPortraitVo>> resultModel = positionPullerIntegration
                .exportPositionUserPortrait(param);
        if (resultModel != null && resultModel.isSuccessed()
                && !CollectionUtils.isEmpty(resultModel.getReturnValue())) {
            List<PositionUserPortraitVo> list = resultModel.getReturnValue();
            //移除合计
            list.remove(0);
            resultModel.setReturnValue(list);
        }
        return resultModel;
    }

    /**
     * @Title: queryPositionClassTypeList  
     * @Description: 查询媒体相关配置列表
     * @param parentId 父ID
     * @param typeCode 所属分类
     * @param name 标签名称
     * @return ResultModel<List<PositionClassType>> 
      */
    @RequestMapping(value = "/mediaConfigList", method = RequestMethod.GET)
    public ResultModel<List<PositionClassType>> queryPositionClassTypeList(
            @RequestParam(value = "parentId", required = false, defaultValue = "0") Long parentId,
            @RequestParam(value = "typeCode") String typeCode,
            @RequestParam(value = "name", required = false) String name) {
        PositionClassTypeDto param = new PositionClassTypeDto();
        param.setParentId(parentId);
        param.setTypeCode(typeCode);
        return mediaCommonConfigFacadeIntegration.queryPositionClassTypeList(param);
    }

    /**
    * @Title: getPositionClassCountList  
    * @Description: 广告位按一级分类统计报表
    * @param className 一级分类名
    * @param datetime 查询时间
    * @return ResultModel<Page<PositionClassCountVo>> 
     */
    @RequestMapping(value = "/position/class", method = RequestMethod.GET)
    public ResultModel<Page<PositionClassCountVo>> getPositionClassCountList(HttpServletRequest request,
            PositionClassCountDto param) {
        if (null == param) {
            param = new PositionClassCountDto();
        }
        return positionPullerIntegration.getPositionClassCountList(param);
    }

    /**
    * @Title: getPositionHourCountList  
    * @Description: 广告位按小时统计报表
    * @param datetime 查询时间
    * @param appName 应用名称
    * @param keywords 广告位ID或者名称
    * @return ResultModel<Page<PositionHourCountVo>> 
     */
    @RequestMapping(value = "/position/hour", method = RequestMethod.GET)
    public ResultModel<Page<PositionHourCountVo>> getPositionHourCountList(HttpServletRequest request, PositionHourCountDto param) {
        if (null == param) {
            param = new PositionHourCountDto();
        }
        if(StringUtils.isNotBlank(param.getAppName()) || StringUtils.isNotBlank(param.getKeywords())){
            QueryAppAdPositionDto queryAppAdPositionDto = new QueryAppAdPositionDto();
            queryAppAdPositionDto.setAppName(param.getAppName());
            queryAppAdPositionDto.setKeywords(param.getKeywords());
            List<AppEntranceAdRo> positionList = appEntranceFacadeIntegration.getPositionList(queryAppAdPositionDto);
            List<String> positionIds = new ArrayList<String>();
            if(!CollectionUtils.isEmpty(positionList)){
                positionList.forEach(position ->{
                    positionIds.add(position.getPositionId());
                });
                param.setPositionIds(positionIds);
            }else{
                param.setPositionIds(positionIds);
            }
        }
        return positionPullerIntegration.getPositionHourCountList(param);
    }

}
