package com.bxm.adscounter.rtb.common;

import com.bxm.adscounter.rtb.common.aop.ControlRtbIntegrationAspect;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.UrlHelper;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Objects;
import java.util.Optional;

/**
 * <p>点击监测追踪器</p>
 * <p>主要处理 RTB 平台的点击监测事件，它提供了保存 {@link #saveClickTracker(KeyValueMap)}、查询 {@link #getClickTracker(String)} 等常用方法。</p>
 * <p>如果这个 RTB 平台启用了 扣量、控量回传、累加回传 等增强功能，那么这个实现类必须也是 {@link RtbIntegration} 的实现，否则不会生效。参考：{@link ControlRtbIntegrationAspect}</p>
 * <p>文档参考：<a href="https://bxmrds.yuque.com/mizhsy/qc669f/wvitbu">RTB-控量回传技术设计方案</a></p>
 *
 * <p>名词解释</p>
 * <p><code>CLICK_ID</code> 是指RTB平台的广告产生点击的唯一标识，类似bxm_id。在点击监测请求、中间页请求里接收，然后在转化时需要回传给RTB。</p>
 *
 * @author allen
 * @date 2022-07-13
 * @since 1.0
 * @see RtbIntegration
 * @see ControlRtbIntegrationAspect
 */
public interface ClickTracker {

    String EMPTY_APP = "-";
    String EMPTY_AD_GROUP_ID = "0";
    String NULL_STRING = "NULL";

    /**
     * RTB 平台
     * @return RTB
     */
    Rtb rtb();

    /**
     * 返回在互动行为日志中 Referrer 参数值里的 <code>CLICK_ID</code> 的参数名。
     * @return <code>CLICK_ID</code>参数名。默认返回 <code>click_id</code>
     */
    default String getClickIdParameterNameOnReferrer() {
        return Common.Param.CLICK_ID;
    }

    /**
     * <p>从点击监测请求参数里找到本次<code>CLICK_ID</code>值。默认获取参数名 <code>click_id</code> 的值。
     * 大部分情况点击监测链接的值是由宏替换的，参数名保持固定。比如快手：{url}?click_id=__CALLBACK__、
     * 百度：{url}?click_id=%%CLICKID%%。所以默认参数名：<code>click_id</code></p>
     * <p>同样的，我们也建议在设计新的RTB点击监测时也保持这样的格式。</p>
     *
     * @param clickTrackerKeyValueMap 日志
     * @return clickId <code>CLICK_ID</code>的值
     */
    default String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    /**
     * <p>从互动广告点击行为日志里找到本次<code>CLICK_ID</code>值。一般是从 Referrer 里获取。</p>
     * <p>默认的实现：先获取{@link #getClickIdParameterNameOnReferrer()} 参数名，如果<code>null</code>则返回<code>null</code>。
     * 否则从<code>clickEventLog</code>中获取refer的值，再将refer的值按URL的格式取出参数名的值。</p>
     *
     * @param clickEventLog 日志
     * @return clickId <code>CLICK_ID</code>值。
     */
    default String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        String clickIdParamNameOnReferrer = getClickIdParameterNameOnReferrer();
        if (Objects.isNull(clickIdParamNameOnReferrer)) {
            return null;
        }
        String ref = clickEventLog.getRef();
        return UrlHelper.getFirstValueOfParamName(ref, clickIdParamNameOnReferrer);
    }

    /**
     * <p>是否强制保存点击监测数据</p>
     *
     * <p>非强制情况下，只有当开启了回传控制，才会保存点击监测数据。</p>
     *
     * @return 默认：false
     */
    default boolean forceSaveClickTracker() {
        return false;
    }

    /**
     * 保存点击监测请求的数据
     *
     * @param clickTrackerKeyValueMap 日志
     */
    void saveClickTracker(KeyValueMap clickTrackerKeyValueMap);

    /**
     * 获取指定 <code>CLICK_ID</code> 的点击监测数据
     * @param clickId CLICK_ID
     * @return 点击监测数据
     */
    KeyValueMap getClickTracker(String clickId);

    /**
     * <p>从回传请求查找 <code>adGroupId</code>。</p>
     *
     * <p>一般情况下，回传请求的 KeyValueMap 不会直接携带 <code>adGroupId</code>，而是需要从 Referrer 中找到 clickId，
     * 然后再通过 {@link #getClickTracker(String)} 方法来获取 <code>adGroupId</code>。</p>
     *
     * <p><i>回传请求的 KeyValueMap 实际是从变现猫广告点击回填的日志。</i></p>
     *
     * <p>下面默认的实现代码：</p>
     * <pre>
     *     &#064;Override
     *     public String getAdGroupId(FeedbackRequest request) {
     *         String referrer = request.getReferrer();
     *         UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
     *         MultiValueMap&lt;String, String&gt; queryParams = build.getQueryParams();
     *         String clickId = request.getClickId();
     *         if (StringUtils.isBlank(clickId)) {
     *             clickId = queryParams.getFirst(getClickIdParamNameOnReferrer());
     *         }
     *
     *         KeyValueMap clickTracker = getClickTracker(clickId);
     *
     *         return Optional.ofNullable(clickTracker)
     *                 .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
     *                 .orElse(StringUtils.EMPTY);
     *     }
     * </pre>
     *
     * @param request 回传请求
     * @return adGroupId
     */
    default String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = request.getClickId();
        if (StringUtils.isBlank(clickId)) {
            clickId = queryParams.getFirst(getClickIdParameterNameOnReferrer());
        }

        KeyValueMap clickTracker = getClickTracker(clickId);

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    /**
     * <p>从指定行为中查找媒体标识</p>
     *
     * @param keyValueMap 行为日志
     * @return app
     */
    default String getApp(KeyValueMap keyValueMap) {
        return EMPTY_APP;
    }

    /**
     * 创建点击监测的存储 Key
     * @param clickId clickId
     * @return KeyGenerator
     */
    default KeyGenerator createKey(String clickId) {
        Rtb rtb = rtb();
        return () -> KeyBuilder.build("rtb", "click", "tracker", rtb.getType(), clickId);
    }

    /**
     * 为了rtb控制针对空adGroupId处理，返回特定的空值数组
     * 如 百度的空值数组为 {"__UNIT_ID__", "NULL"}
     * 如果不需要处理，返回null即可
     * @return 逻辑空值数组
     */
    default String[] getMissingAdGroupIdList() {
        return null;
    }

    /**
     * 当 getMissingAdGroupIdList() 返回不为空时
     * 如果 <code>adGroupId} 无效的，那么返回默认值。
     * 无效是指：空的、获取包含在逻辑空值数组里。
     * @param adGroupId 指定值
     * @return 默认值：{@link #EMPTY_AD_GROUP_ID}
     * @see #getMissingAdGroupIdList()
     */
    default String fixAdGroupIdIfInvalid(String adGroupId) {
        if (StringUtils.isBlank(adGroupId) || StringUtils.equalsIgnoreCase(adGroupId, NULL_STRING)) {
            return ClickTracker.EMPTY_AD_GROUP_ID;
        }
        String[] missingAdGroupIdList = getMissingAdGroupIdList();
        if (null != missingAdGroupIdList) {
            if (ArrayUtils.contains(missingAdGroupIdList, adGroupId)) {
                return ClickTracker.EMPTY_AD_GROUP_ID;
            }
        }
        return adGroupId;
    }
 }
