package com.bxm.adscounter.rtb.common.aop.interceptor;

import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.RtbProperties;
import com.bxm.adscounter.rtb.common.aop.RtbFeedbackInterceptor;
import com.bxm.adscounter.rtb.common.feedback.ActionType;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.feedback.SmartConvType;
import com.bxm.adscounter.rtb.common.service.AbstractFeedbackService;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.extension.client.HttpClientOpenLogClient;
import com.bxm.openlog.extension.client.HttpClientUtils;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.Production;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.IOException;
import java.util.Objects;
import java.util.Optional;

/**
 * @author allen
 * @date 2022-05-13
 * @since 1.0
 */
@Slf4j
public class RtbConversionOpenLogInterceptor implements RtbFeedbackInterceptor {

    private final RtbProperties properties;
    private final OpenLogClient openLogClient;

    public RtbConversionOpenLogInterceptor(RtbProperties properties, OpenLogClient openLogClient) {
        this.properties = properties;
        this.openLogClient = openLogClient;
    }

    @Override
    public void doRequest(RtbIntegration integration, FeedbackRequest request) {
    }

    @Override
    public void doSuccess(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response) {
        this.sendOpenLog(integration, request, response, 1, null);
    }

    @Override
    public void doFail(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response) {
        this.sendOpenLog(integration, request, response, 2, null);
    }

    @Override
    public void doException(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response, Throwable throwable) {
        this.sendOpenLog(integration, request, response, 3, throwable);
    }

    private void sendOpenLog(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response, int rtbSc, Throwable throwable) {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        PositionRtb positionRtb = request.getConfig();
        KeyValueMap map = new KeyValueMap();

        String msg = "";
        if (Objects.nonNull(throwable)) {
            String name = "";
            if (throwable instanceof RtbIntegrationException) {
                RtbIntegrationException e = (RtbIntegrationException) throwable;
                name = e.getFailType().name();
            }
            msg = StringUtils.substring(name + " - " + throwable.getMessage(), 0, 255);
        }

        map.put(Common.RtbParam.P, Production.COMMON.getName());
        map.put(Common.RtbParam.MT, Common.Mt.rtbConversion.original());
        map.put(Common.RtbParam.BXMID, keyValueMap.getFirst(Inads.Param.BXMID));
        map.put(Common.RtbParam.TYPE, keyValueMap.getFirst(Inads.Param.CONVERSION_TYPE));
        map.put(Common.RtbParam.UID, keyValueMap.getFirst(Inads.Param.UID));
        map.put(Common.RtbParam.TIME, System.currentTimeMillis());
        map.put(Common.RtbParam.ADID, keyValueMap.getFirst(Inads.Param.ADID));
        map.put(Common.RtbParam.TAGID, keyValueMap.getFirst(Inads.Param.TAGID));
        map.put(Common.RtbParam.CREATEID, keyValueMap.getFirst(Inads.Param.CREATEID));
        map.put(Common.RtbParam.IP, keyValueMap.getFirst(Inads.Param.IP));
        map.put(Common.RtbParam.UA, keyValueMap.getFirst(Inads.Param.UA));
        map.put(Common.Param.RTB_ID, integration.rtb().getType());
        map.put(Common.Param.AD_GROUP_ID, request.getAdGroupId());
        map.put(Common.RtbParam.RTB_STATUS, rtbSc);
        map.putIfNotBlank(Common.RtbParam.REFER, request.getReferrer());
        map.putIfNotBlank(Common.RtbParam.RTB_REQUEST_URI, request.getRequestUrl());
        map.putIfNotBlank(Common.RtbParam.RTB_EVENT_TYPE, request.getEventType());
        map.putIfNotBlank(Common.RtbParam.RTB_REQ_BODY, request.getRequestBody());
        map.putIfNotBlank(Common.RtbParam.RTB_RES_BODY, Optional.ofNullable(response).map(FeedbackResponse::getBody).orElse(StringUtils.EMPTY));
        map.putIfNotBlank(Common.RtbParam.RTB_RES_MSG, msg);
        map.putIfNotBlank(Common.RtbParam.IMEI_MD5, keyValueMap.getFirst(Inads.Param.IMEI_MD5));
        map.putIfNotBlank(Common.RtbParam.IDFA_MD5, keyValueMap.getFirst(Inads.Param.IDFA_MD5));
        map.putIfNotBlank(Common.RtbParam.ANDROIDID_MD5, keyValueMap.getFirst(Inads.Param.ANDROIDID_MD5));
        map.putIfNotBlank(Common.RtbParam.OAID, keyValueMap.getFirst(Inads.Param.OAID));
        map.putIfNotBlank(Common.RtbParam.OAID_MD5, keyValueMap.getFirst(Inads.Param.OAID_MD5));

        map.putIfNotBlank(Common.RtbParam.ACTION_TIME, keyValueMap.getFirst(Inads.Param.TIME));
        map.putIfNotBlank(Common.RtbParam.ACTION_TYPE, Optional.ofNullable(request.getActionType()).map(ActionType::getValue).orElse(null));
        map.putIfNotBlank(Common.RtbParam.SMART_CONV_TYPE, Optional.ofNullable(request.getSmartConvType()).map(SmartConvType::getValue).orElse(SmartConvType.NONE.getValue()));
        map.putIfNotBlank(Common.RtbParam.RANK_BY_ACTIVITY, AbstractFeedbackService.getAcquiredRankByActivity(keyValueMap));
        map.putIfNotBlank(Common.RtbParam.RETRY_FLAG, request.getRetryTimes() > 0 ? "1" : "0");

        if (positionRtb != null) {
            map.put(Common.RtbParam.FEEDBACK_MODE, positionRtb.getFeedbackMode());
        }

        String openLogRequestUri = map.createOpenLogRequestUri(properties.getOpenLogRequestDomain(), true);

        if (Objects.equals(PositionRtb.AllSourceType.UC, integration.rtb().getType())) {
            log.info("UC debug: {}", openLogRequestUri);
        }
        openLogClient.asyncRequest(openLogRequestUri);
    }

    public static void main(String[] args) throws IOException {
        HttpClientOpenLogClient clientOpenLogClient = HttpClientOpenLogClient.builder().build();

        String testUrl = "http://openlog-inner.bianxianmao.com/common/s.gif?p={abc}";
        new HttpGet(testUrl);
    }
}
