package com.bxm.adscounter.rtb.common.aop.interceptor;

import com.bxm.adscounter.model.constant.RedisKeyGenerator;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.aop.RtbFeedbackInterceptor;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.util.Strings;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.Duration;
import java.util.Objects;

/**
 * @author tangxiao
 * @date 2024-04-02
 * @since 1.0
 */
@Slf4j
public class SaveClickIdInterceptor implements RtbFeedbackInterceptor {

    private static final int EXPIRE_TIME = Math.toIntExact(Duration.ofDays(7).getSeconds());

    private final JedisPool jedisPool;

    public SaveClickIdInterceptor(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    @Override
    public void doRequest(RtbIntegration integration, FeedbackRequest request) {
        saveClickId(request);
    }

    @Override
    public void doSuccess(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response) {
    }

    @Override
    public void doFail(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response) {
    }

    @Override
    public void doException(RtbIntegration integration, FeedbackRequest request, FeedbackResponse response, Throwable throwable) {
    }

    private void saveClickId(FeedbackRequest request) {
        PositionRtb config = request.getConfig();
        if (Objects.isNull(config)) {
            return;
        }
        String positionId = config.getPositionId();
        String clickId = request.getClickId();
        if (StringUtils.isBlank(clickId)) {
            String rtbExt = request.getKeyValueMap().getFirst("rtb_ext");
            if (StringUtils.isNotBlank(rtbExt)) {
                RtbExtObject rtbExtObj = JsonHelper.convert(UrlHelper.urlDecode(rtbExt), RtbExtObject.class);
                clickId = rtbExtObj.getClick_id();
            }
        }
        if (StringUtils.isNotBlank(clickId)) {
            addClickId(positionId, clickId);
        }
    }

    private void addClickId(String positionId, String clickId) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = RedisKeyGenerator.strFeedbackClickId(positionId, clickId).generateKey();
            jedis.set(RedisKeyGenerator.strFeedbackClickId(positionId, clickId).generateKey(), Strings.EMPTY);
            jedis.expire(key, EXPIRE_TIME);
        }
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }


    @Data
    public static class RtbExtObject {
        private String click_id;
    }
}
