package com.bxm.adscounter.rtb.common.control;

import com.bxm.warcar.zk.ZkClientHolder;
import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.recipes.locks.InterProcessLock;
import org.apache.curator.framework.recipes.locks.InterProcessMutex;
import redis.clients.jedis.JedisPool;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 基于分布式的控制调度器，跨进程下只有一个实例允许执行。
 *
 * @author allen
 * @date 2022-07-14
 * @since 1.0
 */
@Slf4j
public abstract class AbstractDistributedControlScheduler implements Runnable {

    private final JedisPool jedisPool;
    private InterProcessLock lock;

    public AbstractDistributedControlScheduler(JedisPool jedisPool, ZkClientHolder zkClientHolder,
                                               String dimension, String namespace) {
        this.jedisPool = jedisPool;
        if (Objects.nonNull(zkClientHolder)) {
            String path = namespace + dimension;
            this.lock = new InterProcessMutex(zkClientHolder.get(), path);
        }
    }

    /**
     * 执行任务
     */
    protected abstract void doRun();

    @Override
    public void run() {
        boolean acquired = true;
        try {
            if (Objects.nonNull(lock)) {
                acquired = lock.acquire(1, TimeUnit.NANOSECONDS);
            }
            if (acquired) {
                doRun();
            }
        } catch (Exception e) {
            log.error("doRun: ", e);
        } finally {
            if (Objects.nonNull(lock) && acquired) {
                try {
                    lock.release();
                } catch (Exception e) {
                    log.error("release: ", e);
                }
            }
        }
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }
}
