package com.bxm.adscounter.rtb.common.control;

import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.RtbIntegrationFactory;
import com.bxm.adscounter.rtb.common.control.plus.PlusQueueService;
import com.bxm.adscounter.rtb.common.control.plus.PlusControl;
import com.bxm.adscounter.rtb.common.control.plus.PlusControlConfig;
import com.bxm.adscounter.rtb.common.control.plus.RedisPlusControlImpl;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.mapper.SrcAdUserAccessLogMapper;
import com.bxm.adscounter.rtb.common.service.PositionRtbService;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.SafeMapHelper;
import com.bxm.warcar.utils.TypeHelper;
import com.bxm.warcar.zk.ZkClientHolder;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import redis.clients.jedis.JedisPool;

import java.time.LocalDate;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author tangxiao
 * @date 2024-03-28
 * @since 1.0
 */
@Slf4j
public abstract class AbstractPlusControlRtbIntegration implements PlusControlRtbIntegration, DisposableBean {

    private final ScheduledThreadPoolExecutor refreshExecutor = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("refresh"));
    private final ConcurrentHashMap<String /* 控制器 Key (Dimension) */, PlusControl> controlExecutors = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Integer /* 配置ID */, Set<String> /* 控制器 Key (Dimension) */> mapping = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String /* createReferencedKey */, String /* 控制器 Key (Dimension) */ > referenced = new ConcurrentHashMap<>();
    private final JedisPool jedisPool;
    private final PositionRtbService service;
    private final EventPark eventPark;
    private final SrcAdUserAccessLogMapper srcAdUserAccessLogMapper;
    private final PlusQueueService plusQueueService;
    private MeterRegistry registry;
    private ZkClientHolder zkClientHolder;
    private RtbIntegrationFactory rtbIntegrationFactory;

    public AbstractPlusControlRtbIntegration(JedisPool jedisPool, PositionRtbService service, EventPark eventPark, RtbIntegrationFactory rtbIntegrationFactory, SrcAdUserAccessLogMapper srcAdUserAccessLogMapper, PlusQueueService plusQueueService) {
        this.jedisPool = jedisPool;
        this.service = service;
        this.eventPark = eventPark;
        this.srcAdUserAccessLogMapper = srcAdUserAccessLogMapper;
        this.rtbIntegrationFactory = rtbIntegrationFactory;
        this.plusQueueService = plusQueueService;
        this.refreshExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.refreshControlForMapping();
            } catch (Exception e) {
                log.error("occur ex: ", e);
            }
        }, 30, 30, TimeUnit.SECONDS);
    }

    @Autowired(required = false)
    public void setZkClientHolder(ZkClientHolder zkClientHolder) {
        this.zkClientHolder = zkClientHolder;
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
    }

    @Override
    public void destroy() {
        this.refreshExecutor.shutdownNow();
    }

    private List<PositionRtb.PlusControl> chooseControls(PositionRtb config) {
        return config.getPlusControls();
    }

    private void refreshControlForMapping() {
        if (log.isDebugEnabled()) {
            log.debug("Starting refresh Plus Controller...");
        }

        createForEnablePositions();

        Set<String> alive = new HashSet<>(referenced.values());
        mapping.forEach(new BiConsumer<Integer, Set<String>>() {
            @Override
            public void accept(Integer id, Set<String> keys) {
                keys.forEach(key -> {
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] Checking {}", id, key);
                    }
                    String tagId = ControlUtils.splitKey(key)[0];
                    PositionRtb config = service.get(tagId);
                    if (Objects.isNull(config)) {
                        // 配置无效，关闭
                        close(id);
                        return;
                    }

                    List<PositionRtb.PlusControl> controls = config.getPlusControls();
                    if (CollectionUtils.isEmpty(controls)) {
                        // 控制配置无效，关闭
                        close(id);
                        return;
                    }
                    if (!alive.contains(key)) {
                        // 已经没有引用，关闭
                        close(id);
                        return;
                    }

                    for (PositionRtb.PlusControl cfg : controls) {
                        if (!Objects.equals(id, cfg.getId())) {
                            continue;
                        }
                        // 状态无效，关闭
                        if (!cfg.isEnable()) {
                            close(id);
                            continue;
                        }

                        // 配置时间超过7天，关闭
                        if (!validDate(cfg.getDate())) {
                            close(id);
                            continue;
                        }

                        refresh(id, cfg, config);
                    }
                });
            }
        });
    }

    private void refresh(Integer id, PositionRtb.PlusControl cfg, PositionRtb positionRtb) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.forEach(key -> {
                PlusControl control = controlExecutors.get(key);
                if (Objects.nonNull(control)) {
                    PlusControlConfig config = control.getConfig();
                    Integer limit = cfg.getLimit();
                    if (config.isUpdate(limit)) {
                        config.setLimit(limit);
                        config.setPositionRtb(positionRtb);
                        control.refresh();
                        if (log.isInfoEnabled()) {
                            log.info("{} Update to: {}", key, config.toSimpleString());
                        }
                    }
                }
            });
        }
    }

    private void close(Integer id) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.removeIf(new Predicate<String>() {
                @Override
                public boolean test(String key) {
                    PlusControl control = controlExecutors.get(key);
                    if (Objects.nonNull(control)) {
                        control.delete();
                        control.shutdown();
                    }
                    controlExecutors.remove(key);
                    if (log.isInfoEnabled()) {
                        log.info("Close control: {} - {}", key, control);
                    }
                    return true;
                }
            });
        }
    }

    /**
     * 为所有可用的广告位补量配置创建控制器
     */
    private void createForEnablePositions() {
        List<PositionRtb> enablePositionRtbs = service.getAll().stream()
                .filter(p -> CollectionUtils.isNotEmpty(p.getPlusControls()))
                .collect(Collectors.toList());

        for (PositionRtb config : enablePositionRtbs) {
            String tagId = config.getPositionId();
            List<PositionRtb.PlusControl> controls = chooseControls(config);
            if (CollectionUtils.isNotEmpty(controls)) {
                for (PositionRtb.PlusControl cfg : controls) {
                    if (!cfg.isEnable()) {
                        continue;
                    }
                    // 配置时间超过7天不创建
                    if (!validDate(cfg.getDate())) {
                        continue;
                    }

                    createIfNecessary(config.getSourceType(), tagId, cfg.getAdGroupId(), cfg, config);
                }
            }
        }
    }

    private void createIfNecessary(Integer sourceType, String tagId, String adGroupId, PositionRtb.PlusControl hit, PositionRtb config) {
        String key = ControlUtils.createKey(tagId, adGroupId, TypeHelper.castToString(hit.getId()));
        SafeMapHelper.get(controlExecutors, key, new SafeMapHelper.InitializingValue<PlusControl>() {
            @Override
            public PlusControl initializing() {
                Rtb rtb = Rtb.of(sourceType);
                RtbIntegration instance = rtbIntegrationFactory.get(rtb);
                String date = hit.getDate();
                String startTime = hit.getStartTime();
                String endTime = hit.getEndTime();
                PlusControlConfig plusControlConfig = new PlusControlConfig()
                        .setDimension(key)
                        .setControlId(hit.getId())
                        .setDate(date)
                        // 这里日期时间合并了
                        .setStartTime(date + " " + startTime)
                        .setEndTime(date + " " + endTime)
                        .setLimit(hit.getLimit())
                        .setN(hit.getN())
                        .setY(hit.getY())
                        .setPositionId(tagId)
                        .setAdGroupId(adGroupId)
                        .setPositionRtb(config)
                        .setConsumer(new Consumer<FeedbackRequest>() {
                            @Override
                            public void accept(FeedbackRequest request) {
                                doFeedbackQuietly(instance, request);
                            }
                        });
                PlusControl plusControl = new RedisPlusControlImpl(plusControlConfig, jedisPool, registry, eventPark, zkClientHolder, srcAdUserAccessLogMapper, plusQueueService);
                plusControl.start();
                SafeMapHelper.get(mapping, hit.getId(), (SafeMapHelper.InitializingValue<Set<String>>) HashSet::new).add(key);
                // 保存引用
                String referencedKey = tagId + adGroupId + hit.getId();
                String previous = referenced.put(referencedKey, key);
                if (Objects.nonNull(previous)) {
                    log.info("Plus control {} reference changed to {}", previous, key);
                }
                return plusControl;
            }
        });
    }

    private void doFeedbackQuietly(RtbIntegration instance, FeedbackRequest request) {
        try {
            instance.doFeedback(request, 1);
        } catch (RtbIntegrationException e) {
            log.error("doFeedbackQuietly: ", e);
        }
    }

    private static boolean validDate(String date) {
        LocalDate localDate = LocalDateTimeUtils.parseDate(date);
        LocalDate sevenDayAgo = LocalDate.now().minusDays(7);
        return localDate != null && localDate.isAfter(sevenDayAgo);
    }

}
