package com.bxm.adscounter.rtb.common.control;

import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.RtbIntegrationFactory;
import com.bxm.adscounter.rtb.common.control.rate.RateControl;
import com.bxm.adscounter.rtb.common.control.rate.RateControlConfig;
import com.bxm.adscounter.rtb.common.control.rate.RedisRateControlImpl;
import com.bxm.adscounter.rtb.common.feedback.ActionType;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.mapper.SrcAdUserAccessLogMapper;
import com.bxm.adscounter.rtb.common.service.PositionRtbService;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.SafeMapHelper;
import com.bxm.warcar.zk.ZkClientHolder;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import redis.clients.jedis.JedisPool;

import java.time.LocalDate;
import java.time.LocalTime;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.bxm.adscounter.rtb.common.control.LocalDateTimeUtils.parseDate;
import static com.bxm.adscounter.rtb.common.control.LocalDateTimeUtils.parseTime;

/**
 * @author allen
 * @date 2022-07-14
 * @since 1.0
 */
@Slf4j
@Deprecated
public abstract class AbstractRateControlRtbIntegration implements RateControlRtbIntegration, DisposableBean {

    private final ScheduledThreadPoolExecutor refreshExecutor = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("refresh"));
    private final ConcurrentHashMap<String /* 控制器 Key (Dimension) */, RateControl> controlExecutors = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long /* 配置ID */, Set<String> /* 控制器 Key (Dimension) */> mapping = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String /* createReferencedKey */, String /* 控制器 Key (Dimension) */ > referenced = new ConcurrentHashMap<>();
    private final JedisPool jedisPool;
    private final PositionRtbService service;
    private final EventPark eventPark;
    private final SrcAdUserAccessLogMapper srcAdUserAccessLogMapper;
    private MeterRegistry registry;
    private ZkClientHolder zkClientHolder;
    private RtbIntegrationFactory rtbIntegrationFactory;

    public AbstractRateControlRtbIntegration(JedisPool jedisPool, PositionRtbService service, EventPark eventPark, RtbIntegrationFactory rtbIntegrationFactory, SrcAdUserAccessLogMapper srcAdUserAccessLogMapper) {
        this.jedisPool = jedisPool;
        this.service = service;
        this.eventPark = eventPark;
        this.srcAdUserAccessLogMapper = srcAdUserAccessLogMapper;
        this.rtbIntegrationFactory = rtbIntegrationFactory;
        this.refreshExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.refreshControlForMapping();
            } catch (Exception e) {
                log.error("occur ex: ", e);
            }
        }, 1, 1, TimeUnit.MINUTES);
    }

    @Autowired(required = false)
    public void setZkClientHolder(ZkClientHolder zkClientHolder) {
        this.zkClientHolder = zkClientHolder;
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
    }

    @Override
    public void destroy() {
        this.refreshExecutor.shutdownNow();
    }

    /**
     * 从指定配置里，找到当前 速率控制 的配置项目。
     * @param config 配置
     * @return 速率控制配置项目。
     */
    protected abstract List<PositionRtb.SpeedControl> chooseControls(PositionRtb config);

    /**
     * 创建控制器的 Key。
     *
     * @param tagId 广告位
     * @param adId 广告券ID，如果是活动行为这个值为空
     * @param adGroupId 广告组
     * @param cfg 控制配置
     * @return Key
     */
    protected abstract String createControlKey(String tagId, String adId, String adGroupId, PositionRtb.SpeedControl cfg);

    /**
     * 创建控制器引用的 Key。
     *
     * @param tagId 广告位
     * @param adId 广告券ID，如果是活动行为这个值为空
     * @param adGroupId 广告组
     * @param cfg 控制配置
     * @return Key
     */
    protected abstract String createReferencedKey(String tagId, String adId, String adGroupId, PositionRtb.SpeedControl cfg);

    private void refreshControlForMapping() {
        if (log.isDebugEnabled()) {
            log.debug("Starting refresh Rate Controller...");
        }

        createForEnablePositions();

        Set<String> alive = new HashSet<>(referenced.values());
        mapping.forEach(new BiConsumer<Long, Set<String>>() {
            @Override
            public void accept(Long id, Set<String> keys) {
                keys.forEach(key -> {
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] Checking {}", id, key);
                    }
                    String tagId = ControlUtils.splitKey(key)[0];
                    PositionRtb config = service.get(tagId);
                    if (Objects.isNull(config)) {
                        // 配置无效，关闭
                        close(id);
                        return;
                    }

                    List<PositionRtb.SpeedControl> controls = chooseControls(config);
                    if (CollectionUtils.isEmpty(controls)) {
                        // 控制配置无效，关闭
                        close(id);
                        return;
                    }
                    if (!alive.contains(key)) {
                        // 已经没有引用，关闭
                        close(id);
                        return;
                    }

                    for (PositionRtb.SpeedControl cfg : controls) {
                        if (!Objects.equals(id, cfg.getId())) {
                            continue;
                        }
                        // 状态无效，关闭
                        if (!cfg.isEnable()) {
                            close(id);
                            continue;
                        }
                        // 时间无效，关闭
                        boolean validTime = isValidTime(cfg.getDate(), cfg.getStartTime(), cfg.getEndTime());
                        if (!validTime) {
                            close(id);
                            continue;
                        }

                        refresh(id, cfg, config);
                    }
                });
            }
        });
    }

    private void refresh(Long id, PositionRtb.SpeedControl cfg, PositionRtb positionRtb) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.forEach(key -> {
                RateControl control = controlExecutors.get(key);
                if (Objects.nonNull(control)) {
                    RateControlConfig config = control.getConfig();
                    Integer count = cfg.getCount();
                    LocalTime start = parseTime(cfg.getStartTime());
                    LocalTime end = parseTime(cfg.getEndTime());
                    if (config.isUpdate(count, start, end)) {
                        config.setCount(count);
                        config.setStart(start);
                        config.setEnd(end);
                        config.setPositionRtb(positionRtb);
                        control.refresh();
                        if (log.isInfoEnabled()) {
                            log.info("{} Update to: {}", key, config.toSimpleString());
                        }
                    }
                }
            });
        }
    }

    private void close(Long id) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.removeIf(new Predicate<String>() {
                @Override
                public boolean test(String key) {
                    RateControl control = controlExecutors.get(key);
                    if (Objects.nonNull(control)) {
                        control.delete();
                        control.shutdown();
                    }
                    controlExecutors.remove(key);
                    if (log.isInfoEnabled()) {
                        log.info("Close control: {} - {}", key, control);
                    }
                    return true;
                }
            });
        }
    }

    /**
     * 为所有可用的广告位控量配置创建控制器
     */
    private void createForEnablePositions() {
        List<PositionRtb> enablePositionRtbs = service.getAll().stream()
                .filter(p -> CollectionUtils.isNotEmpty(p.getTicketControls()) || CollectionUtils.isNotEmpty(p.getActivityControls()))
                .collect(Collectors.toList());

        for (PositionRtb config : enablePositionRtbs) {
            String tagId = config.getPositionId();
            List<PositionRtb.SpeedControl> controls = chooseControls(config);
            if (CollectionUtils.isNotEmpty(controls)) {
                for (PositionRtb.SpeedControl cfg : controls) {
                    if (!cfg.isEnable()) {
                        continue;
                    }
                    boolean validTime = isValidTime(cfg.getDate(), cfg.getStartTime(), cfg.getEndTime());
                    if (!validTime) {
                        continue;
                    }
                    createIfNecessary(config.getSourceType(), tagId, cfg.getAdTicketId(), cfg.getAdGroupId(), cfg, config);
                }
            }
        }
    }

    private RateControl createIfNecessary(Integer sourceType, String tagId, String adId, String adGroupId, PositionRtb.SpeedControl hit, PositionRtb config) {
        String key = createControlKey(tagId, adId, adGroupId, hit);
        return SafeMapHelper.get(controlExecutors, key, new SafeMapHelper.InitializingValue<RateControl>() {
            @Override
            public RateControl initializing() {
                Rtb rtb = Rtb.of(sourceType);
                RtbIntegration instance = rtbIntegrationFactory.get(rtb);
                LocalTime start = parseTime(hit.getStartTime());
                LocalTime end = parseTime(hit.getEndTime());
                RateControlConfig rateControlConfig = new RateControlConfig()
                        .setDimension(key)
                        .setStart(start)
                        .setEnd(end)
                        .setCount(hit.getCount())
                        .setPositionId(tagId)
                        .setAdGroupId(adGroupId)
                        .setPositionRtb(config)
                        .setActionType(getActionType(hit))
                        .setAdId(hit.getAdTicketId())
                        .setConsumer(new Consumer<FeedbackRequest>() {
                            @Override
                            public void accept(FeedbackRequest request) {
                                doFeedbackQuietly(instance, request);
                            }
                        });
                RateControl rateControl = new RedisRateControlImpl(rateControlConfig, jedisPool, registry, eventPark, zkClientHolder, srcAdUserAccessLogMapper);
                rateControl.start();
                SafeMapHelper.get(mapping, hit.getId(), (SafeMapHelper.InitializingValue<Set<String>>) HashSet::new).add(key);
                // 保存引用
                String referencedKey = createReferencedKey(tagId, adId, adGroupId, hit);
                String previous = referenced.put(referencedKey, key);
                if (Objects.nonNull(previous)) {
                    log.info("Rate control {} reference changed to {}", previous, key);
                }
                return rateControl;
            }
        });
    }

    protected boolean isValidTime(String date, String startTime, String endTime) {
        LocalDate d = parseDate(date);
        LocalTime start = parseTime(startTime);
        LocalTime end = parseTime(endTime);
        if (Objects.isNull(d) || Objects.isNull(start) || Objects.isNull(end)) {
            return false;
        }
        boolean isToday = LocalDate.now().isEqual(d);
        LocalTime now = LocalTime.now().withNano(0);
        // start >= now <= end
        boolean isValidTime = (now.equals(start) || now.equals(end) || (now.isAfter(start) && now.isBefore(end)));
        return isToday && isValidTime;
    }

    protected abstract ActionType getActionType(PositionRtb.SpeedControl cfg);

    private void doFeedbackQuietly(RtbIntegration instance, FeedbackRequest request) {
        try {
            instance.doFeedback(request, 1);
        } catch (RtbIntegrationException e) {
            log.error("doFeedbackQuietly: ", e);
        }
    }

}
