package com.bxm.adscounter.rtb.common.control;

import com.bxm.adscounter.rtb.common.ClickTracker;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationFactory;
import com.bxm.adscounter.rtb.common.control.plus.PlusQueueService;
import com.bxm.adscounter.rtb.common.control.cpa.CpaControl;
import com.bxm.adscounter.rtb.common.control.deduction.ConversionLevel;
import com.bxm.adscounter.rtb.common.control.ratio.*;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioAdClickEvent;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioClickEvent;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioConversionEvent;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.service.PositionRtbService;
import com.bxm.adscounter.rtb.common.utils.PositionRTBUtils;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.SafeMapHelper;
import com.bxm.warcar.utils.TypeHelper;
import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.collect.Sets;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.LocalTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.function.ToDoubleFunction;

import static com.bxm.adscounter.rtb.common.control.LocalDateTimeUtils.parseTime;

/**
 * 控制回传转化。
 *
 * @author tangxiao
 * @date 2022-08-05
 * @since 1.0
 */
@Slf4j
public class DefaultRatioControlRtbIntegrationImpl implements RatioControlRtbIntegration, DisposableBean {

    private final ScheduledThreadPoolExecutor refreshExecutor = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("refresh"));
    private final ConcurrentHashMap<String /* 控制器 Key (Dimension) */, RatioControl> controlExecutors = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Long /* 配置ID */, Set<String> /* 控制器 Key (Dimension) */> mapping = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String /* 广告位 + adGroupId */, String /* 控制器 Key (Dimension) */ > referenced = new ConcurrentHashMap<>();
    private final JedisPool jedisPool;
    private final PositionRtbService service;
    private final EventPark eventPark;
    private final CpaControl cpaControl;
    private final PlusQueueService plusQueueService;
    private MeterRegistry registry;
    private ZkClientHolder zkClientHolder;
    private RtbIntegrationFactory factory;

    public DefaultRatioControlRtbIntegrationImpl(JedisPool jedisPool, PositionRtbService service, EventPark eventPark, CpaControl cpaControl, PlusQueueService plusQueueService) {
        this.jedisPool = jedisPool;
        this.service = service;
        this.eventPark = eventPark;
        this.cpaControl = cpaControl;
        this.plusQueueService = plusQueueService;
        this.refreshExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.refreshControlForMapping();
            } catch (Exception e) {
                log.error("occur ex: ", e);
            }
        }, 1, 1, TimeUnit.MINUTES);
    }

    @Autowired
    public void setZkClientHolder(ZkClientHolder zkClientHolder) {
        this.zkClientHolder = zkClientHolder;
    }

    @Autowired
    public void setFactory(RtbIntegrationFactory factory) {
        this.factory = factory;
    }

    @Override
    public void destroy() {
        this.refreshExecutor.shutdownNow();
        this.controlExecutors.values().forEach(e -> {
            e.delete();
            e.shutdown();
        });
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
        this.registryMeter();
    }

    private void registryMeter() {
        // 任务数
        this.registry.gauge("ratio.control.executors", 0, new ToDoubleFunction<Integer>() {
            @Override
            public double applyAsDouble(Integer value) {
                return controlExecutors.size();
            }
        });
    }

    private boolean isHeadTicket(PositionRtb config, String adid) {
        return PositionRTBUtils.isHeadTicket(adid, config);
    }

    @Override
    public void onClick(RtbIntegration instance, KeyValueMap clickTrackerKeyValueMap, String adGroupId) {
        this.onClick(instance, clickTrackerKeyValueMap, adGroupId, null);
    }

    @Override
    public void onClick(RtbIntegration instance, KeyValueMap clickTrackerKeyValueMap, String adGroupId, String app) {
        app = StringUtils.defaultIfBlank(app, ClickTracker.EMPTY_APP);

        if (!(instance instanceof ClickTracker)) {
            return;
        }
        ClickTracker clickTracker = (ClickTracker) instance;
        adGroupId = clickTracker.fixAdGroupIdIfInvalid(adGroupId);

        if (StringUtils.isBlank(adGroupId)) {
            return;
        }

        String tagId = clickTrackerKeyValueMap.getFirst(Inads.Param.TAGID);
        if (StringUtils.isBlank(tagId)) {
            log.warn("tagid is blank for {}!", adGroupId);
            return;
        }
        PositionRtb config = service.get(tagId);
        if (Objects.isNull(config)) {
            return;
        }
        PositionRtb.RatioControl hit = chooseCvrControl(config, adGroupId);
        if (Objects.nonNull(hit)) {
            String clickId = (clickTracker).getClickId(clickTrackerKeyValueMap);
            if (isRepeatClick(clickTracker, clickId)) {
                log.debug("Repeat click: {}", clickId);
                return;
            }
            RatioControl control = createIfNecessary(instance, tagId, adGroupId, hit, config);
            control.onClick(clickId, app);
            eventPark.post(new RatioClickEvent(this, control, adGroupId, app));
        }
    }

    @Override
    public void onTicketClick(RtbIntegration instance, KeyValueMap clickTrackerKeyValueMap, KeyValueMap clickKeyValueMap, String adGroupId, String app) {
        app = StringUtils.defaultIfBlank(app, ClickTracker.EMPTY_APP);
        if (StringUtils.isBlank(adGroupId)) {
            return;
        }
        if (!(instance instanceof ClickTracker)) {
            return;
        }
        ClickTracker clickTracker = (ClickTracker) instance;
        String tagId = clickKeyValueMap.getFirst(Inads.Param.TAGID);
        String adid = clickKeyValueMap.getFirst(Inads.Param.ADID);
        if (StringUtils.isBlank(tagId)) {
            return;
        }
        PositionRtb config = service.get(tagId);
        if (Objects.isNull(config)) {
            return;
        }
        if (!config.isCvrOrDeductionControl()) {
            return;
        }
        PositionRtb.RatioControl hit = chooseCvrControl(config, adGroupId);
        if (Objects.isNull(hit)) {
            return;
        }
        String clickId = (clickTracker).getClickId(clickTrackerKeyValueMap);

        RatioControl control = createIfNecessary(instance, tagId, adGroupId, hit, config);
        boolean headTicket = isHeadTicket(config, adid);

        ClickLog clickLog = ClickLog.builder()
                .adid(adid)
                .clickId(clickId)
                .app(app)
                .config(config)
                .clickKeyValueMap(clickKeyValueMap)
                .isHeadTicket(headTicket)
                .build();
        control.onAdClick(clickLog);
        eventPark.post(new RatioAdClickEvent(this, control, adGroupId, app));
    }

    @Override
    public boolean onFeedbackControl(RtbIntegration instance, FeedbackRequest request) {
        String adGroupId = request.getAdGroupId();
        if (StringUtils.isBlank(adGroupId)) {
            return false;
        }
        if (!(instance instanceof ClickTracker)) {
            return false;
        }
        ClickTracker clickTracker = (ClickTracker) instance;
        KeyValueMap keyValueMap = request.getKeyValueMap();
        String tagId = keyValueMap.getFirst(Inads.Param.TAGID);
        PositionRtb config = request.getConfig();
        if (Objects.isNull(config)) {
            return false;
        }
        PositionRtb.RatioControl hit = chooseCvrControl(config, adGroupId);
        if (Objects.isNull(hit)) {
            return false;
        }
        String app = StringUtils.defaultIfBlank(clickTracker.getApp(keyValueMap), ClickTracker.EMPTY_APP);
        // 填充app，后面回传时需要用到
        request.setAppid(app);
        // 控制器
        RatioControl control = createIfNecessary(instance, tagId, adGroupId, hit, config);
        // 暂存回传数据
        control.pushConversion(request, app);
        eventPark.post(new RatioConversionEvent(this, control, adGroupId, app));
        return true;
    }

    /**
     * 指定 clickId 是否重复的。
     * @param clickId ClickID
     * @return 返回 true 则表示在一定时间内是重复的。
     */
    private boolean isRepeatClick(ClickTracker clickTracker, String clickId) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = KeyBuilder.build("rtb", "conv", "CLICK", clickTracker.rtb().getType(), clickId);
            Long rs = jedis.incr(key);
            jedis.expire(key, 3600);
            return rs > 1;
        }
    }

    private void refreshControlForMapping() {
        if (log.isDebugEnabled()) {
            log.debug("Starting refresh Ratio Controller...");
        }
        Set<String> alive = new HashSet<>(referenced.values());

        Set<Long> removeIf = Sets.newHashSet();

        mapping.forEach((id, keys) -> {
            keys.forEach(key -> {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Checking {}", id, key);
                }
                String tagId = ControlUtils.splitKey(key)[0];
                PositionRtb config = service.get(tagId);
                if (Objects.isNull(config)) {
                    // 配置无效时，关闭
                    removeIf.add(id);
                    return;
                }

                List<PositionRtb.RatioControl> controls = config.getRatioControls();
                if (CollectionUtils.isEmpty(controls)) {
                    // 控制配置无效时，关闭
                    removeIf.add(id);
                    return;
                }

                if (!config.isCvrOrDeductionControl()) {
                    // 不是cvr回传，关闭
                    removeIf.add(id);
                    return;
                }

                if (!alive.contains(key)) {
                    // 已经没有引用，关闭
                    removeIf.add(id);
                    return;
                }

                for (PositionRtb.RatioControl cfg : controls) {
                    if (!Objects.equals(id, cfg.getId())) {
                        continue;
                    }
                    if (!cfg.isEnable()) {
                        // 状态关闭时，关闭
                        removeIf.add(id);
                        continue;
                    }
                    String startTime = cfg.getStartTime();
                    String endTime = cfg.getEndTime();
                    if (!isValidTime(startTime, endTime)) {
                        // 无效时间，关闭
                        removeIf.add(id);
                        continue;
                    }

                    // 刷新配置
                    refresh(id, config, cfg);
                }
            });
        });

        removeIf.forEach(this::close);

        removeIf.forEach(mapping::remove);

        if (log.isInfoEnabled()) {
            log.info("Reference executor: {}, Need remove: {}, After remove on Mapping: {}", alive.size(), removeIf.size(), mapping.size());
        }
    }

    private void refresh(Long id, PositionRtb rtbConfig, PositionRtb.RatioControl cfg) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.forEach(key -> {
                RatioControl control = controlExecutors.get(key);
                if (Objects.nonNull(control)) {
                    RatioControlConfig config = control.getConfig();
                    RatioControlConfig newConfig = createRatioControlConfig(config.getDimension(), rtbConfig, cfg, config.getTagId(), config.getAdGroupId());
                    control.refreshConfig(newConfig);
                }
            });
        }
    }

    private void close(Long id) {
        Set<String> executors = mapping.get(id);
        if (CollectionUtils.isNotEmpty(executors)) {
            executors.removeIf(new Predicate<String>() {
                @Override
                public boolean test(String key) {
                    RatioControl control = controlExecutors.get(key);
                    if (Objects.nonNull(control)) {
                        control.delete();
                        control.shutdown();
                    }
                    controlExecutors.remove(key);
                    if (log.isInfoEnabled()) {
                        log.info("{} - Close control: {} - {}", id, key, control);
                    }
                    return true;
                }
            });
        }
    }

    private RatioControl createIfNecessary(RtbIntegration instance, String tagId, String adGroupId, PositionRtb.RatioControl hit,
                                           PositionRtb config) {
        String key = ControlUtils.createKey(tagId, adGroupId, TypeHelper.castToString(hit.getId()), hit.getStartTime(), hit.getEndTime());
        return SafeMapHelper.get(controlExecutors, key, new SafeMapHelper.InitializingValue<RatioControl>() {
                    @Override
                    public RatioControl initializing() {
                        // 有命中的控制条目
                        RatioControlConfig ratioControlConfig = createRatioControlConfig(key, config, hit, tagId, adGroupId);
                        RatioControlBus bus = RatioControlBus.builder()
                                .instance(instance)
                                .jedisPool(jedisPool)
                                .registry(registry)
                                .eventPark(eventPark)
                                .zkClientHolder(zkClientHolder)
                                .factory(factory)
                                .positionRtbService(service)
                                .build();
                        RatioControl ratioControl = new RedisRatioControlImpl(bus, ratioControlConfig, cpaControl, plusQueueService);
                        ratioControl.start();
                        // 保存当前配置对应的控制器 Key
                        SafeMapHelper.get(mapping, hit.getId(), (SafeMapHelper.InitializingValue<Set<String>>) HashSet::new).add(key);
                        // 保存引用
                        String previous = referenced.put(tagId + adGroupId, key);
                        if (Objects.nonNull(previous)) {
                            log.info("Ratio control {} reference changed to {}", previous, key);
                        }
                        return ratioControl;
                    }
                });
    }

    private RatioControlConfig createRatioControlConfig(String dimension, PositionRtb config, PositionRtb.RatioControl hit, String tagId, String adGroupId) {
        RatioControlConfig ratioControlConfig = new RatioControlConfig();

        PositionRtb.ControlStrategy controlStrategy = config.getControlStrategy();
        if (Objects.nonNull(controlStrategy)) {
            RatioControlConfig.Duration2 duration2 = RatioControlConfig.Duration2.of(controlStrategy.getDuration());
            if (Objects.nonNull(duration2)) {
                ratioControlConfig.setDuration(duration2);
            }
            RatioControlConfig.PlusStrategy strategy = RatioControlConfig.PlusStrategy.of(controlStrategy.getPlusStrategy());
            if (Objects.nonNull(strategy)) {
                ratioControlConfig.setPlusStrategy(strategy);
            }
        }

        return ratioControlConfig
                .setTagId(tagId)
                .setAdvertiserId(config.getCustomerId())
                .setAdGroupId(adGroupId)
                .setDimension(dimension)
                .setHitConfigId(hit.getId())
                .setRatio(Optional.ofNullable(hit.getCvr()).orElse(0d))
                .setDeductionShallowRatio(Optional.ofNullable(hit.getShallowRatio()).orElse(0d))
                .setDeductionDeepRatio(Optional.ofNullable(hit.getDeepRatio()).orElse(0d))
                .setCost(Optional.ofNullable(hit.getCost()).orElse(0d))
                .setExpireInHours(2 * 24)
                .setDataFetchInMinute(5)
                .setShallowEventType(config.getTargetOneRtb())
                .setDeepEventType(config.getTargetTwoRtb())
                .setEnableCvrControl(hit.isEnableCvrControl())
                .setEnableDeductionControl(hit.isEnableDeductionControl())
                .setEnableCostControl(hit.isEnableCostControl())
                .setConversionLevel(Optional.ofNullable(config.getTargetType()).filter(t -> t == 2).map(t -> ConversionLevel.DEEP).orElse(ConversionLevel.SHALLOW))
                .setLimitUnconvs(hit.getLimitUnconv())
                .setAdvertiserName(config.getSourceId())
                ;
    }

    private PositionRtb.RatioControl chooseCvrControl(PositionRtb config, String adGroupId) {
        List<PositionRtb.RatioControl> ratioControls = config.getRatioControls();
        if (CollectionUtils.isEmpty(ratioControls)) {
            return null;
        }
        PositionRtb.RatioControl hit = null;
        for (PositionRtb.RatioControl ratioControl : ratioControls) {
            if (!ratioControl.isEnable()) {
                continue;
            }
            String theAdGroupId = ratioControl.getAdGroupId();
            // start >= now <= end
            String startTime = ratioControl.getStartTime();
            String endTime = ratioControl.getEndTime();
            if (!isValidTime(startTime, endTime)) {
                continue;
            }
            if (StringUtils.isBlank(theAdGroupId)) {
                // 全局定向
                hit = ratioControl;
            }
            if (StringUtils.equals(adGroupId, theAdGroupId)) {
                // [优先] 广告组定向
                hit = ratioControl;
                break;
            }
        }
        return hit;
    }

    private boolean isValidTime(String startTime, String endTime) {
        LocalTime start = parseTime(startTime);
        LocalTime end = parseTime(endTime);
        if (Objects.isNull(start) || Objects.isNull(end)) {
            return false;
        }
        LocalTime now = LocalTime.now().withNano(0);
        // start >= now <= end
        return (now.equals(start) || now.equals(end) || (now.isAfter(start) && now.isBefore(end)));
    }
}
