package com.bxm.adscounter.rtb.common.control.cpa;

import com.alibaba.fastjson.JSON;
import com.bxm.adscounter.rtb.common.*;
import com.bxm.adscounter.rtb.common.data.AdGroupData;
import com.bxm.adscounter.rtb.common.data.DorisAdGroupConversionData;
import com.bxm.adscounter.rtb.common.data.DorisAdGroupHourlyData;
import com.bxm.adscounter.rtb.common.data.Parameter;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.SmartConvType;
import com.bxm.adscounter.rtb.common.impl.kuaishou.KuaishouDataFetcher;
import com.bxm.adscounter.rtb.common.mapper.DorisMapper;
import com.bxm.adscounter.rtb.common.service.PositionRtbService;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.adsprod.facade.ticket.rtb.SummarySmartHostingConfig;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import com.bxm.warcar.message.dingding.DingDingMessageSender;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Getter;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.scheduling.annotation.Scheduled;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 *
 *
 * @author tangxiao
 * @date 2023-10-10
 * @since 1.0
 */
@Slf4j
public class SummarySmartHostingScheduler {
    private final PositionRtbService positionRtbService;
    private final KuaishouDataFetcher kuaishouDataFetcher;
    private final DorisMapper dorisMapper;
    private final CpaControl cpaControl;
    private final RtbIntegrationFactory rtbIntegrationFactory;
    private final Updater updater;
    private final Counter counter;
    private final MessageSender messageSender;

    public SummarySmartHostingScheduler(PositionRtbService positionRtbService,
                                        KuaishouDataFetcher kuaishouDataFetcher,
                                        DorisMapper dorisMapper,
                                        CpaControl cpaControl,
                                        RtbIntegrationFactory rtbIntegrationFactory,
                                        Updater updater,
                                        Counter counter,
                                        RtbProperties properties) {
        this.positionRtbService = positionRtbService;
        this.kuaishouDataFetcher = kuaishouDataFetcher;
        this.dorisMapper = dorisMapper;
        this.cpaControl = cpaControl;
        this.rtbIntegrationFactory = rtbIntegrationFactory;
        this.updater = updater;
        this.counter = counter;

        this.messageSender = new DingDingMessageSender(properties.getNotifyDingtalkUrl());
    }

    /**
     * 累加回传-智能托管-调度执行
     *
     */
    @Scheduled(cron = "0 10,20,30,40,50,55 * * * ? ")
    public void execute() {
        log.info("【smart hosting】执行开始！");
        long start = System.currentTimeMillis();
        // 获取所有广告位rtb配置
        List<PositionRtb> availablePositionRtb = positionRtbService.getAll()
                .stream()
                .filter(PositionRtb::isEnableSmartHosting)
                .collect(Collectors.toList());

        for (PositionRtb positionRtb : availablePositionRtb) {
            String positionId = positionRtb.getPositionId();
            try {
                if (log.isInfoEnabled()) {
                    log.info("[{}] rtb-summary smart hosting scheduler started.................", positionId);
                }

                List<SummarySmartHostingConfig.Config> adGroupConf = getAdGroupConf(positionRtb);
                if (CollectionUtils.isEmpty(adGroupConf)) {
                    continue;
                }

                List<ComputeParameter> computeParameters;

                try {
                    computeParameters = getComputeParameterList(positionRtb, adGroupConf);
                } catch (Exception e) {
                    log.error("getComputeParameterList occur error", e);
                    continue;
                }

                if (log.isInfoEnabled()) {
                    log.info("[{}] computeParameters : {}", positionId, JSON.toJSONString(computeParameters));
                }

                if (CollectionUtils.isEmpty(computeParameters)) {
                    continue;
                }

                // 后台智能托管配置中的广告组id
                Set<String> confAdGroupIds = adGroupConf.stream()
                        .map(SummarySmartHostingConfig.Config::getAdGroupId)
                        .filter(adGroupId -> !isAccountDimension(adGroupId))
                        .collect(Collectors.toSet());

                RtbIntegration integration = rtbIntegrationFactory.get(Rtb.of(positionRtb.getSourceType()));

                // 补量、切换算法
                smartHostingPlusAndSwitch(computeParameters, confAdGroupIds, integration, positionId);

                // 存储数据
                saveFetchInfo(computeParameters, positionId);

                if (log.isInfoEnabled()) {
                    log.info("[{}] rtb-summary smart hosting scheduler end!", positionId);
                }
            } catch (Exception e) {
                log.error(String.format("[%s] execute occur error", positionId) , e);
            }
        }

        log.info("【smart hosting】本次调度执行完成，耗时：{}ms", System.currentTimeMillis() - start);
    }

    private void saveFetchInfo(List<ComputeParameter> computeParameters, String positionId) {
        for (ComputeParameter computeParameter : computeParameters) {
            for (SmartHostingInfoFieldEnum fieldEnum : SmartHostingInfoFieldEnum.values()) {
                updater.hupdate(hashAdGroupHourInfo(positionId, computeParameter.getAdGroupId()),
                        fieldEnum.getFieldName(),
                        Optional.ofNullable(fieldEnum.getFetchFunction().apply(computeParameter)).orElse(fieldEnum.getDefaultValue()));
                updater.expire(hashAdGroupHourInfo(positionId, computeParameter.getAdGroupId()), 24 * 60 * 60);
            }
        }
    }

    private void smartHostingPlusAndSwitch(List<ComputeParameter> computeParameters, Set<String> confAdGroupIds, RtbIntegration integration, String positionId) {
        for (ComputeParameter computeParameter : computeParameters) {
            List<String> activeAdGroupIds = getActiveAdGroupIds(computeParameter, confAdGroupIds);
            if (CollectionUtils.isEmpty(activeAdGroupIds)) {
                continue;
            }

            // 先存下队列中的可补数
            updater.hupdate(hashAdGroupHourInfo(positionId, computeParameter.adGroupId), "plusQueueLength", cpaControl.getConversionQueueLength(positionId, activeAdGroupIds));

            if (computeParameter.getShouldIntercept()) {
                continue;
            }
            if (computeParameter.getEnableBreaker()) {
                continue;
            }

            int plusCount = computeParameter.getNeedPlusCount();
            if (plusCount <= 0) {
                continue;
            }

            List<FeedbackRequest> feedbackRequests = cpaControl.getOnConversionQueue(positionId, activeAdGroupIds, plusCount);

            // 当补量数不足，设置为智能托管算法
            if (plusCount > feedbackRequests.size()) {
                computeParameter.setEnableSmartHosting(true);
                for (String activeAdGroupId : activeAdGroupIds) {
                    switchSmartHosting(positionId, activeAdGroupId);
                }
            }

            for (FeedbackRequest feedbackRequest : feedbackRequests) {
                try {
                    log.info("[{} - {}] 智能托管补量回传中...", positionId, computeParameter.getAdGroupId());
                    feedbackRequest.setSmartConvType(SmartConvType.SMART_HOSTING_QUEUE);
                    integration.doFeedback(feedbackRequest, 1);
                } catch (RtbIntegrationException e) {
                    log.error("RTB smart hosting. plus feedback occur exception", e);
                } finally {
                    // 补量统计
                    counter.hincrementAndGet(hashAdGroupHourInfo(positionId, computeParameter.adGroupId), "plusNum");
                }
            }
        }
    }

    /**
     * 获取实际生效的广告组
     * 账户配置：除后台配置的其他广告组
     * 广告组配置：原广告组
     * @param computeParameter
     * @param confAdGroupIds
     * @return
     */
    private List<String> getActiveAdGroupIds(ComputeParameter computeParameter, Set<String> confAdGroupIds) {
        if (isAccountDimension(computeParameter.getAdGroupId())) {
            List<String> activeAdGroupIds = computeParameter.getFetchKuaishouAdGroupIds().stream()
                    .filter(kuaishouAdGroupId -> !confAdGroupIds.contains(kuaishouAdGroupId))
                    .collect(Collectors.toList());
            // 账户维度加上0的广告组，便于后续切换算法
            activeAdGroupIds.add(ClickTracker.EMPTY_AD_GROUP_ID);
            return activeAdGroupIds;
        }
        return Lists.newArrayList(computeParameter.getAdGroupId());
    }

    private void switchSmartHosting(String positionId, String adGroupId) {
        Long summaryCpaBalance = cpaControl.getSummaryCpaBalance(positionId, adGroupId);
        // 累加cpa余额为0不用切换
        if (summaryCpaBalance == 0) {
            return;
        }
        updater.update(strAlgSmartHostingSwitch(positionId, adGroupId), 1, Math.toIntExact(Duration.ofMinutes(15).getSeconds()));

        sendDingDing(String.format("广告位:%s，广告组:%s 切换为智能托管算法!", positionId, adGroupId));
    }

    private List<ComputeParameter> getComputeParameterList(PositionRtb positionRtb, List<SummarySmartHostingConfig.Config> adGroupConf) {
        Parameter parameter = Parameter.builder()
                .advertiserId(positionRtb.getCustomerId())
                .tagId(positionRtb.getPositionId())
                .needFetchKuaishouCpa(true)
                .build();

        List<AdGroupData> kuaishouAdGroupDataList;
        try {
            kuaishouAdGroupDataList = kuaishouDataFetcher.fetchCurrentHourData(parameter);
            if (Objects.isNull(kuaishouAdGroupDataList)) {
                throw new DataFetchFailException(String.format("kuaishou fetch data is null. positionId: %s", positionRtb.getPositionId()));
            }
        } catch (DataFetchFailException e) {
            log.warn("data fetch fail : {}", e.getMessage());
            return null;
        }

        Map<String, AdGroupData> kuaishouAdGroupDataMap = kuaishouAdGroupDataList.stream().filter(data ->  StringUtils.isNotBlank(data.getAdGroupId())).collect(Collectors.toMap(AdGroupData::getAdGroupId, d -> d));
        List<DorisAdGroupHourlyData> adGroupHourlyData = dorisMapper.getAdGroupHourlyData(positionRtb.getPositionId(), null, DateHelper.format(DateHelper.PATTERN_STR10), DateHelper.format("HH"));
        List<DorisAdGroupConversionData> adGroupConversionData = dorisMapper.getAdGroupConversionData(positionRtb.getPositionId(), null, DateHelper.format(DateHelper.PATTERN_STR8), getHour());
        Map<String, Long> dorisAdGroupConversionMap = adGroupConversionData.stream().collect(Collectors.toMap(data -> data.getAdGroupId(), data -> data.getSendValidPv()));
        Map<String, BigDecimal> dorisAdGroupIncomeMap = adGroupHourlyData.stream().collect(Collectors.toMap(data -> data.getAdGroupId(), data -> data.getIncome()));

        log.info("positionId: {},  kuaishou list :{}", positionRtb.getPositionId(), JSON.toJSONString(kuaishouAdGroupDataList));
        log.info("positionId: {},  doris income list :{}", positionRtb.getPositionId(), JSON.toJSONString(adGroupHourlyData));
        log.info("positionId: {},  doris conversion list :{}", positionRtb.getPositionId(), JSON.toJSONString(adGroupConversionData));

        return adGroupConf.stream()
                .map(adGroupConfig -> buildComputeParameter(adGroupConfig, kuaishouAdGroupDataList, kuaishouAdGroupDataMap, dorisAdGroupConversionMap, dorisAdGroupIncomeMap, positionRtb))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private ComputeParameter buildComputeParameter(SummarySmartHostingConfig.Config config, List<AdGroupData> kuaishouAdGroupDataList, Map<String, AdGroupData> kuaishouAdGroupDataMap, Map<String, Long> dorisAdGroupConversionMap, Map<String, BigDecimal> dorisAdGroupIncomeMap, PositionRtb positionRtb) {
        String adGroupId = config.getAdGroupId();
        BigDecimal charge = Optional.ofNullable(getKuaishouCharge(adGroupId, kuaishouAdGroupDataList, kuaishouAdGroupDataMap)).orElse(BigDecimal.ZERO);
        BigDecimal cpa = Optional.ofNullable(getCpa(config, kuaishouAdGroupDataMap, adGroupId)).orElse(BigDecimal.ZERO);
        Double minRoi = config.getMinRoi();
        Double excessRatio = config.getExcessRatio();
        BigDecimal income = Optional.ofNullable(getDorisIncome(adGroupId, dorisAdGroupIncomeMap)).orElse(BigDecimal.ZERO);
        Long dorisConversionNum = Optional.ofNullable(getDorisConversionNum(adGroupId, dorisAdGroupConversionMap)).orElse(0L);
        Long kuaishouConversionNum = Optional.ofNullable(getKuaishouConversionNum(adGroupId, kuaishouAdGroupDataList, kuaishouAdGroupDataMap)).orElse(0L);

        log.info("[{} - {}] charge: {}, income: {}, dorisConversionNum: {}, kuaishouConversionNum: {}，cpa: {}", positionRtb.getPositionId(), config.getAdGroupId(), charge, income, dorisConversionNum, kuaishouConversionNum, cpa);

        Long activeConversionNum = getActiveConversionNum(kuaishouConversionNum, dorisConversionNum);
        BigDecimal roi = computeRoi(income, charge);
        boolean roiIntercept = isRoiIntercept(roi, minRoi);
        BigDecimal billCost = computeBillCost(charge, activeConversionNum);
        boolean billCostIntercept = isBillCostIntercept(billCost, excessRatio, cpa);
        Integer plusCount = computePlusCount(charge, excessRatio, cpa, activeConversionNum);
        boolean shouldIntercept = roiIntercept || billCostIntercept;
        boolean enableBreaker = isEnableBreaker(dorisConversionNum, kuaishouConversionNum, income, charge);

        return new ComputeParameter()
                .setAdGroupId(adGroupId)
                .setShouldIntercept(shouldIntercept)
                .setNeedPlusCount(plusCount)
                .setCharge(charge)
                .setConversionNum(activeConversionNum)
                .setBillCost(billCost)
                .setRoi(roi)
                .setEnableBreaker(enableBreaker)
                .setFetchKuaishouAdGroupIds(getFetchKuaishouAdGroupIds(kuaishouAdGroupDataList, adGroupId))
                .setBillCostCpaFormula(getBillCostCpaFormula(billCost, cpa));
    }

    private boolean isBillCostIntercept(BigDecimal billCost, Double excessRatio, BigDecimal cpa) {
        if (billCost == null) {
            return true;
        }
        boolean billCostIntercept = billCost.compareTo(BigDecimal.valueOf(excessRatio).multiply(cpa)) <= 0;
        log.info("billCostIntercept({}) = {} <= {} * {}", billCostIntercept, billCost, excessRatio, cpa);
        return billCostIntercept;
    }

    private boolean isRoiIntercept(BigDecimal roi, Double minRoi) {
        if (roi == null) {
            return true;
        }
        boolean isRoiIntercept = roi.compareTo(BigDecimal.valueOf(minRoi)) <= 0;
        log.info("roiIntercept({})= {} <= {}", isRoiIntercept, roi, minRoi);
        return isRoiIntercept;
    }

    /**
     * 是否熔断
     * 小时快手和变现猫都有回传数据，回传成功数与快手计费转化数gap绝对值大于5个且比例绝对值超过20%以上都熔断，则本次熔断
     * 当前小时券收入/花费>1.5，则本次熔断
     * @param dorisConversionNum
     * @param kuaishouConversionNum
     * @param income
     * @param charge
     * @return
     */
    private boolean isEnableBreaker(Long dorisConversionNum, Long kuaishouConversionNum, BigDecimal income, BigDecimal charge) {
        long absoluteValue = Math.abs(dorisConversionNum - kuaishouConversionNum);
        double absoluteValueRatio;
        if (dorisConversionNum == 0) {
            return false;
        } else {
            absoluteValueRatio = BigDecimal.valueOf(absoluteValue).divide(BigDecimal.valueOf(dorisConversionNum), 2, RoundingMode.HALF_UP).doubleValue();
        }

        double incomeChargeRatio;

        if (BigDecimal.ZERO.compareTo(charge) == 0) {
            return false;
        } else {
            incomeChargeRatio = income.divide(charge, 6, RoundingMode.HALF_UP).doubleValue();
        }

        boolean enableBreaker = absoluteValue > 5 || absoluteValueRatio > 0.2 || incomeChargeRatio > 1.5;
        log.info("enableBreaker({}) = {} > 5 || {} > 0.2 || {} > 1.5", enableBreaker, absoluteValue, absoluteValueRatio, incomeChargeRatio);

        return enableBreaker;
    }

    private List<String> getFetchKuaishouAdGroupIds(List<AdGroupData> kuaishouAdGroupDataList, String confAdGroupId) {
        if (isAccountDimension(confAdGroupId)) {
            return kuaishouAdGroupDataList.stream().map(AdGroupData::getAdGroupId).collect(Collectors.toList());
        }
        return null;
    }

    private BigDecimal getCpa(SummarySmartHostingConfig.Config config, Map<String, AdGroupData> kuaishouAdGroupDataMap, String adGroupId) {
        if (isAccountDimension(adGroupId)) {
            return BigDecimal.valueOf(config.getCpa());
        }
        AdGroupData kuaishouData = kuaishouAdGroupDataMap.get(adGroupId);
        return Optional.ofNullable(kuaishouData).map(AdGroupData::getCpa).orElse(null);
    }

    private BigDecimal getKuaishouCharge(String adGroupId, List<AdGroupData> kuaishouAdGroupData, Map<String, AdGroupData> kuaishouAdGroupDataMap) {
        if (isAccountDimension(adGroupId)) {
            if (kuaishouAdGroupData.isEmpty()) {
                return null;
            }
            return kuaishouAdGroupData.stream()
                    .map(AdGroupData::getCharge)
                    .reduce(BigDecimal::add)
                    .orElse(null);
        }
        AdGroupData data = kuaishouAdGroupDataMap.get(adGroupId);
        return Optional.ofNullable(data).map(AdGroupData::getCharge).orElse(null);
    }

    private Long getKuaishouConversionNum(String adGroupId, List<AdGroupData> kuaishouAdGroupData, Map<String, AdGroupData> kuaishouAdGroupDataMap) {
        if (isAccountDimension(adGroupId)) {
            if (kuaishouAdGroupData.isEmpty()) {
                return null;
            }
            return kuaishouAdGroupData.stream()
                    .map(AdGroupData::getConvNumByImpression)
                    .reduce(Long::sum)
                    .orElse(null);
        }
        AdGroupData data = kuaishouAdGroupDataMap.get(adGroupId);
        return Optional.ofNullable(data).map(AdGroupData::getConvNumByImpression).orElse(null);
    }

    private BigDecimal getDorisIncome(String adGroupId, Map<String, BigDecimal> dorisAdGroupIncomeMap) {
        if (isAccountDimension(adGroupId)) {
            if (dorisAdGroupIncomeMap.isEmpty()) {
                return null;
            }
            return dorisAdGroupIncomeMap.values()
                    .stream()
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        return dorisAdGroupIncomeMap.getOrDefault(adGroupId, null);
    }

    /**
     * MAPI获取的转化数和我们自己的时段回传成功数对比，gap绝对值大于5个则以我们的回传数为准，否则按获取的数据为准
     * @param kuaishouConversionNum
     * @param dorisConversionNum
     * @return
     */
    private Long getActiveConversionNum(Long kuaishouConversionNum, Long dorisConversionNum) {
        boolean condition = Math.abs(kuaishouConversionNum - dorisConversionNum) > 5;
        if (condition) {
            return dorisConversionNum;
        }
        return kuaishouConversionNum;
    }

    private Integer computePlusCount(BigDecimal charge, Double excessRatio, BigDecimal cpa, Long activeConversionNum) {
        BigDecimal excessRatioCpaMultiply = BigDecimal.valueOf(excessRatio).multiply(cpa);
        if (BigDecimal.ZERO.compareTo(excessRatioCpaMultiply) == 0) {
            return 0;
        }
        long plusCount = charge.divide(excessRatioCpaMultiply, 2, RoundingMode.HALF_UP)
                .subtract(BigDecimal.valueOf(activeConversionNum)).longValue();
        log.info("plusCount({}) = {} / ({} * {} - {})", plusCount, charge, excessRatio, cpa, activeConversionNum);
        return Math.toIntExact(plusCount);
    }

    static KeyGenerator strAlgSmartHostingSwitch(String positionId, String adGroupId) {
        return () -> KeyBuilder.build("AD", "RTB", "SMART", "HOSTING", "SWITCH", positionId , adGroupId);
    }

    static KeyGenerator hashAdGroupHourInfo(String positionId, String adGroupId) {
        return () -> KeyBuilder.build("rtb", "summary_feedback", "smart_hosting_info", getHour(), positionId , adGroupId);
    }

    private static String getHour() {
        return DateHelper.format("yyyyMMddHH");
    }

    /**
     * 计算计费成本
     * @param charge 花费
     * @param conversionNum 转化数
     * @return
     */
    private BigDecimal computeBillCost(BigDecimal charge, Long conversionNum) {
        if (Objects.equals(conversionNum, 0L)) {
            return null;
        }
        BigDecimal billCost = charge.divide(BigDecimal.valueOf(conversionNum), 2, RoundingMode.HALF_UP);
        log.info("billCost({}) = {} / {}", billCost, charge, conversionNum);
        return billCost;
    }

    private BigDecimal computeRoi(BigDecimal income, BigDecimal charge) {
        if (charge.compareTo(BigDecimal.ZERO) == 0) {
            return null;
        }
        BigDecimal roi = income.divide(charge, 3, RoundingMode.HALF_UP);
        log.info("roi（{}) = {} / {}", roi, income, charge);
        return roi;
    }

    @AllArgsConstructor
    @Getter
    enum SmartHostingInfoFieldEnum {
        charge("charge", ComputeParameter::getCharge, StringUtils.EMPTY),
        conversionNum("conversionNum", ComputeParameter::getConversionNum, BigDecimal.ZERO.toString()),
        billCost("billCost", ComputeParameter::getBillCost, "--"),
        roi("roi", ComputeParameter::getRoi, "--"),
        enableSmartHosting("enableSmartHosting", ComputeParameter::getEnableSmartHosting, Boolean.FALSE.toString()),
        enableBreaker("enableBreaker", ComputeParameter::getEnableBreaker, StringUtils.EMPTY),
        billCostCpaFormula("billCostCpaFormula", ComputeParameter::getBillCostCpaFormula, StringUtils.EMPTY),
        ;
        public final String fieldName;
        public final Function<ComputeParameter, Object> fetchFunction;
        public final String defaultValue;
    }

    @Data
    @Accessors(chain = true)
    public static class ComputeParameter {
        String adGroupId;

        /**
         * 花费
         */
        BigDecimal charge;

        /**
         * 计费转化数
         */
        Long conversionNum;

        /**
         * 计费成本
         */
        BigDecimal billCost;

        BigDecimal roi;

        Boolean shouldIntercept;

        /**
         * 待补量的数量
         */
        int needPlusCount;

        Boolean enableSmartHosting;

        List<String> fetchKuaishouAdGroupIds;

        Boolean enableBreaker;

        /**
         * 时段补量统计
         */
        Integer hourPlus;

        /**
         * 计费成本/cpa 计算公式
         */
        String billCostCpaFormula;
    }

    Long getDorisConversionNum(String adGroupId, Map<String, Long> dorisAdGroupConversionMap) {
        if (isAccountDimension(adGroupId)) {
            if (dorisAdGroupConversionMap.isEmpty()) {
                return null;
            }
            return dorisAdGroupConversionMap.values()
                    .stream()
                    .reduce(0L, Long::sum);
        }
        return dorisAdGroupConversionMap.get(adGroupId);
    }

    private boolean isAccountDimension(String adGroupId) {
        return Objects.equals(adGroupId, "ACCOUNT");
    }

    private void sendDingDing(String msg) {
        try {
            log.info("钉钉发送消息：{}", msg);
            messageSender.send(new Message(msg));
        } catch (Exception e) {
            log.error("send: ", e);
        }
    }

    private List<SummarySmartHostingConfig.Config> getAdGroupConf(PositionRtb positionRtb) {
        String summarySmartHostingConf = positionRtb.getSummarySmartHostingConf();
        if (StringUtils.isBlank(summarySmartHostingConf)) {
            return null;
        }
        SummarySmartHostingConfig convert = JsonHelper.convert(summarySmartHostingConf, SummarySmartHostingConfig.class);
        SummarySmartHostingConfig.Config accountConf = convert.getAccountConf();
        List<SummarySmartHostingConfig.Config> adGroupConf = Optional.ofNullable(convert.getAdGroupConf()).orElse(Lists.newArrayList());
        adGroupConf.add(accountConf);
        return adGroupConf;
    }

    private String getBillCostCpaFormula(BigDecimal billCost, BigDecimal cpa) {
        if (billCost == null) {
            return String.format("--/%s=--", cpa);
        }
        return cpa.compareTo(BigDecimal.ZERO) == 0 ? String.format("%s/0=--", billCost) :
                String.format("%s/%s=%s", billCost, cpa, billCost.divide(cpa, 2, RoundingMode.HALF_UP));
    }

}
