package com.bxm.adscounter.rtb.common.control.plus;

import com.bxm.adscounter.model.constant.RedisKeyGenerator;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 补量队列实现
 * @author tangxiao
 * @date 2023/6/8
 * @since 1.0
 */
@Slf4j
public class PlusQueueServiceImpl implements PlusQueueService{

    private static final int EXPIRE_TIME = Math.toIntExact(Duration.ofDays(7).getSeconds());
    private static final String SPLIT = "|";

    private final JedisPool jedisPool;
    public PlusQueueServiceImpl(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    @Override
    public void pushConversion(FeedbackRequest feedbackRequest) {
        PositionRtb config = feedbackRequest.getConfig();
        String adGroupId = feedbackRequest.getAdGroupId();
        String positionId = config.getPositionId();
        String data = JsonHelper.convert(feedbackRequest);
        String largeId = createLargeId(data);
        if (log.isDebugEnabled()) {
            log.debug("存储补量回传数据...{} - {}", positionId, adGroupId);
        }
        String key = listConversionQueue(positionId, adGroupId).generateKey();
        addConversion(key, largeId);
    }

    @Override
    public List<FeedbackRequest> getOnConversionQueue(String positionId, List<String> adGroupIds, Integer size, String startTime, String endTime) {
        if (size <= 0) {
            return Collections.emptyList();
        }
        List<FeedbackRequest> results = new ArrayList<>();
        // 需要重新放入队列的数据
        List<String> repushList = Lists.newArrayList();

        for (String adGroupId : adGroupIds) {
            if (results.size() >= size) {
                break;
            }
            String key = listConversionQueue(positionId, adGroupId).generateKey();
            do {
                String pop = rightPopWithValid(key, startTime, endTime, repushList);
                if (StringUtils.isBlank(pop)) {
                    break;
                }
                FeedbackRequest feedbackRequest = JsonHelper.convert(pop, FeedbackRequest.class);
                if (!existClickId(positionId, feedbackRequest.getClickId())) {
                    feedbackRequest.setAdGroupId(adGroupId);
                    results.add(feedbackRequest);
                }

            } while (results.size() < size);

            for (String item : repushList) {
                addConversion(key, item);
            }
        }
        return results;
    }

    private String rightPopWithValid(String key, String startTime, String endTime, List<String> repushList) {
        String item = rightPop(key);
        if (StringUtils.isBlank(item)) {
            return null;
        }
        String[] largeId = splitLargeId(item);
        long createTime = NumberUtils.toLong(largeId[0]);

        if (isExpireTime(createTime)) {
            return rightPopWithValid(key, startTime, endTime, repushList);
        }
        if (isNotHitTime(createTime, startTime, endTime)) {
            // 如果时间不是范围内的，则需要重新放入缓存
            repushList.add(item);
            return rightPopWithValid(key, startTime, endTime, repushList);
        }
        if (log.isDebugEnabled()) {
            log.debug("pop {}", item);
        }
        return largeId[1];
    }

    static KeyGenerator listConversionQueue(String positionId, String adGroupId) {
        return () -> KeyBuilder.build("rtb", "common_conv_queue", positionId, adGroupId);
    }

    private void addConversion(String key, String data) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            jedis.lpush(key, data);
            jedis.expire(key, EXPIRE_TIME);
        }
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }

    private String rightPop(String key) {
        try (Jedis jedis = getJedisPool().getResource()) {
            return jedis.rpop(key);
        } catch (Exception e) {
            log.error("rpop: {}", e.getMessage());
            return null;
        }
    }

    private static String createLargeId(String data) {
        return System.currentTimeMillis() + SPLIT + data;
    }

    static String[] splitLargeId(String id) {
        int i = id.indexOf(SPLIT);
        if (i == -1) {
            throw new IllegalStateException(String.format("%s is illegal value", id));
        }
        String t = id.substring(0, i);
        String d = id.substring(i + 1);
        return new String[] { t, d };
    }

    private boolean isNotHitTime(Long createTime, String startTime, String endTime) {
        LocalDateTime startDateTime = LocalDateTime.parse(startTime, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        LocalDateTime endDateTime = LocalDateTime.parse(endTime, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        LocalDateTime createDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(createTime), ZoneId.systemDefault());

        return createDateTime.isBefore(startDateTime) || createDateTime.isAfter(endDateTime);
    }

    private boolean isExpireTime(Long createTime) {
        Duration duration = Duration.ofSeconds(EXPIRE_TIME);
        return (System.currentTimeMillis() - createTime) > duration.toMillis();
    }

    private boolean existClickId(String positionId, String clickId) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            Boolean existClickId = jedis.exists(RedisKeyGenerator.strFeedbackClickId(positionId, clickId).generateKey());
            if (existClickId) {
                log.info("Plus Control the clickId has been feedback. {}", clickId);
            }
            return existClickId;
        }
    }
}
