package com.bxm.adscounter.rtb.common.control.plus;

import com.bxm.adscounter.rtb.common.mapper.SrcAdUserAccessLogMapper;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 基于 Redis 实现的补量控制器
 *
 * @author allen
 * @date 2022-07-13
 * @since 1.0
 */
@Slf4j
public class RedisPlusControlImpl implements PlusControl {
    private static final String SPLIT = "|";
    private static final int COUNT_EXPIRE_TIME = Math.toIntExact(Duration.ofDays(30).getSeconds());
    private final PlusControlConfig config;
    private final ScheduledThreadPoolExecutor control = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("control"));
    private final JedisPool jedisPool;
    private final MeterRegistry registry;
    private final EventPark eventPark;

    private final RedisPlusControlScheduler scheduler;
    private final LeaderLatch leaderLatch;
    private final PlusQueueService plusQueueService;

    public RedisPlusControlImpl(PlusControlConfig config, JedisPool jedisPool, MeterRegistry registry, EventPark eventPark, ZkClientHolder zkClientHolder, SrcAdUserAccessLogMapper srcAdUserAccessLogMapper, PlusQueueService plusQueueService) {
        this.eventPark = eventPark;
        this.plusQueueService = plusQueueService;
        Preconditions.checkNotNull(config);
        Preconditions.checkArgument(StringUtils.isNotBlank(config.getDimension()), "dimension cannot be null");
        Preconditions.checkNotNull(config.getConsumer(), "consumer cannot be null");
        long limit = config.getLimit();
        Preconditions.checkArgument(limit > 0 && limit < 10000, "count must > 0 and < 10000");

        this.config = config;
        this.jedisPool = jedisPool;
        this.registry = registry;
        this.scheduler = new RedisPlusControlScheduler(this, jedisPool, eventPark, srcAdUserAccessLogMapper, registry, this.plusQueueService);
        this.leaderLatch = new LeaderLatch(zkClientHolder.get(), "/adscounter/rtb/control/plus/" + config.getDimension());
    }

    @Override
    public void start() {
        this.leaderLatch.addListener(new LeaderLatchListener() {
            @Override
            public void isLeader() {
                Integer delay = Optional.ofNullable(config.getN()).orElse(1);
                if (log.isInfoEnabled()) {
                    log.info("[{}] plus Control Start, Delay time is {} minute", config.getDimension(), delay);
                }
                control.scheduleWithFixedDelay(scheduler, 0, delay, TimeUnit.MINUTES);
            }

            @Override
            public void notLeader() {
            }
        });
        try {
            this.leaderLatch.start();
        } catch (Exception e) {
            log.error("start: ", e);
        }
    }

    @Override
    public void shutdown() {
        try {
            leaderLatch.close();
        } catch (Exception e) {
            log.error("shutdown: ", e);
        }
        control.shutdownNow();
    }

    @Override
    public void count() {
        this.addCount();
        this.registry.counter("plus.control.count", tags()).increment();
    }

    @Override
    public void countAdGroup(String adGroupId) {
        if (config.isEveryGroupDimension()) {
            this.addAdGroupCount(adGroupId);
            this.registry.counter("plus.control.count.adgroup", tags()).increment();
        }
    }


    @Override
    public int getCount() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            return NumberUtils.toInt(jedis.get(string(config.getDimension()).generateKey()));
        }
    }

    @Override
    public void push(String id) {
        this.lpush(id);
        this.registry.counter("plus.control.list", tags()).increment();
    }

    @Override
    public void refresh() {
        this.control.remove(this.scheduler);
        this.start();
    }

    @Override
    public void delete() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String dimension = config.getDimension();
//            String count = string(dimension).generateKey();
            String list = list(dimension).generateKey();
            jedis.del(list);
        }
    }

    @Override
    public PlusControlConfig getConfig() {
        return this.config;
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }

    private void addCount() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = string(config.getDimension()).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, COUNT_EXPIRE_TIME);
            if (log.isDebugEnabled()) {
                log.debug("{} after decrBy {}", key, rs);
            }
        }
    }

    private void addAdGroupCount(String adGroupId) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = strAdGroupCount(config.getDimension(), adGroupId).generateKey();
            Long rs = jedis.incr(key);
            if (log.isDebugEnabled()) {
                log.debug("{} after decrBy {}", key, rs);
            }
        }
    }

    private void lpush(String id) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            jedis.lpush(list(config.getDimension()).generateKey(), createLargeId(id));
        }
    }

    private List<Tag> tags() {
        return Lists.newArrayList(Tag.of("dim", config.getDimension()));
    }

    static String createLargeId(String id) {
        return System.currentTimeMillis() + SPLIT + id;
    }

    static KeyGenerator string(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "plus", "count", dimension);
    }

    static KeyGenerator list(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "plus", "list", dimension);
    }

    private static KeyGenerator strAdGroupCount(String dimension, String adGroupId) {
        return () -> KeyBuilder.build("rtb_control", "plus", "count", dimension, adGroupId);
    }


}
