package com.bxm.adscounter.rtb.common.control.plus;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.model.constant.RedisKeyGenerator;
import com.bxm.adscounter.rtb.common.control.AbstractStandaloneControlScheduler;
import com.bxm.adscounter.rtb.common.control.plus.event.PlusNotEnoughEvent;
import com.bxm.adscounter.rtb.common.data.QueryParam;
import com.bxm.adscounter.rtb.common.data.SrcAdUserAccessLog;
import com.bxm.adscounter.rtb.common.feedback.ActionType;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.SmartConvType;
import com.bxm.adscounter.rtb.common.mapper.SrcAdUserAccessLogMapper;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 基于 Redis 的补量控制器调度器。
 *
 * @author tangxiao
 * @date 2024-04-08
 * @since 1.0
 */
@Slf4j
public class RedisPlusControlScheduler extends AbstractStandaloneControlScheduler {

    private final PlusControl control;
    private final SrcAdUserAccessLogMapper srcAdUserAccessLogMapper;
    private final MeterRegistry registry;
    private final PlusQueueService plusQueueService;


    public RedisPlusControlScheduler(PlusControl control, JedisPool jedisPool, EventPark eventPark, SrcAdUserAccessLogMapper srcAdUserAccessLogMapper, MeterRegistry registry, PlusQueueService plusQueueService) {
        super(null, jedisPool, eventPark);
        this.control = control;
        this.srcAdUserAccessLogMapper = srcAdUserAccessLogMapper;
        this.registry = registry;
        this.plusQueueService = plusQueueService;
    }

    @Override
    protected void doRun() {
        List<FeedbackRequest> conversions = getConversionsByDimension();
        if (CollectionUtils.isNotEmpty(conversions)) {
            doConversion(conversions);
        }
    }

    private List<FeedbackRequest> getConversionsByDimension() {
        PlusControlConfig config = control.getConfig();
        String adGroupId = config.getAdGroupId();
        int current = control.getCount();
        int total = config.getLimit();

        if (config.isEveryGroupDimension()) {
            return getConversionForEveryGroup();
        }

        int remain = getRemain(total, current, config.getY());
        if (remain <= 0) {
            return Lists.newArrayList();
        }

        List<String> adGroupIds;
        if (config.isTotalDimension()) {
            adGroupIds = srcAdUserAccessLogMapper.getAllAdGroupId(config.getPositionId(), config.getStartTime(), config.getEndTime());
        } else {
            adGroupIds = Lists.newArrayList(adGroupId);
        }

        if (CollectionUtils.isEmpty(adGroupIds)) {
            log.info("find adGroupIds is empty. {}", config.getDimension());
            return Lists.newArrayList();
        }

        List<FeedbackRequest> conversions = getConversions(remain, adGroupIds);

        boolean isPlusNotEnough = remain > conversions.size() && config.notInTime(System.currentTimeMillis());
        if (isPlusNotEnough) {
            getEventPark().post(new PlusNotEnoughEvent(this, config, control));
        }
        return conversions;
    }

    private void doConversion(List<FeedbackRequest> conversions) {
        if (log.isInfoEnabled()) {
            log.info("Plus control wait conversion num: {} | dimension:{}", conversions.size(), control.getConfig().getDimension());
        }
        PlusControlConfig config = control.getConfig();
        for (FeedbackRequest conversion : conversions) {
            if (Objects.nonNull(conversion)) {
                try {
                    conversion.setSmartConvType(SmartConvType.CONV_QUEUE);
                    config.getConsumer().accept(conversion);
                } catch (Exception e) {
                    log.error("occur exception | accept: ", e);
                } finally {
                    control.count();
                    control.countAdGroup(conversion.getAdGroupId());
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] progress: {}/{}", config.getDimension(), control.getCount(), config.getLimit());
                    }
                }
            }
        }
    }

    /**
     * 获取回传数据
     * 优先级：扣量存储 > 数据库券点击 > 数据库活动参与
     * @param remain 需要的回传数
     * @param adGroupId 广告组id
     * @return
     */
    private List<FeedbackRequest> getConversions(Integer remain, List<String> adGroupId) {
        PlusControlConfig config = control.getConfig();
        PositionRtb positionRtbConfig = config.getPositionRtb();
        String positionId = positionRtbConfig.getPositionId();
        List<FeedbackRequest> result = Lists.newArrayList();

        List<FeedbackRequest> resultFromPlus = plusQueueService.getOnConversionQueue(positionId, adGroupId, remain, config.getStartTime(), config.getEndTime());
        log.info("[{}]Plus Queue Need size {} , actual {} adGroupId:{}", config.getDimension(), remain, resultFromPlus.size(), JSON.toJSONString(adGroupId));
        result.addAll(resultFromPlus);
        remain = remain - resultFromPlus.size();

        if (remain > 0) {
            // 从数据库点击中获取
            config.setActionType(ActionType.TICKET_CLICK);
            List<FeedbackRequest> resultFromClick = getConversionsFromDB(positionRtbConfig, config, remain, adGroupId);
            log.info("[{}]DB Click Need size {} , actual {}  adGroupId:{}", config.getDimension(), remain, resultFromClick.size(), JSON.toJSONString(adGroupId));
            result.addAll(resultFromClick);
            remain = remain - resultFromClick.size();
            // 如果再不够，从数据库活动参与中获取
            if (remain > 0) {
                config.setActionType(ActionType.ACTIVITY_ATTEND);
                List<FeedbackRequest> resultFromActivityAttend = getConversionsFromDB(positionRtbConfig, config, remain, adGroupId);
                result.addAll(resultFromActivityAttend);
                log.info("[{}] DB Activity Need size {} , actual {}  adGroupId:{}", config.getDimension(), remain, resultFromActivityAttend.size(), JSON.toJSONString(adGroupId));
            }
        }
        return result;
    }

    /**
     * 如果y为空，则取剩余数
     * @param total 需要达到的数量
     * @param current 当前已回传数量
     * @param y （定时任务）每次需要回传的数量
     * @return 需要回传数量
     */
    private int getRemain(int total, int current, Integer y) {
        int remain = total - current;
        y = Optional.ofNullable(y).orElse(remain);
        return Math.min(remain, y);
    }

    private List<FeedbackRequest> getConversionForEveryGroup() {
        PlusControlConfig config = control.getConfig();
        // 从数据库查出时段所有的广告组id
        List<String> allAdGroupId = srcAdUserAccessLogMapper.getAllAdGroupId(config.getPositionId(), config.getStartTime(), config.getEndTime());

        List<FeedbackRequest> result = Lists.newArrayList();
        int totalCount = 0;
        for (String adGroupId : allAdGroupId) {// 当前组id的回传数
            int count = getEveryAdGroupIdCount(config.getDimension(), adGroupId);
            totalCount += count;
            int remain = getRemain(config.getLimit(), count, config.getY());
            if (remain <= 0) {
                continue;
            }
            List<FeedbackRequest> conversions = getConversions(remain, Lists.newArrayList(adGroupId));
            result.addAll(conversions);
        }

        int totalLimit = config.getLimit() * allAdGroupId.size();
        int count = totalCount + result.size();

        // 判定存在一个广告组达不到数量
        if (config.notInTime(System.currentTimeMillis()) && (totalLimit == 0 || totalLimit > count)) {
            getEventPark().post(new PlusNotEnoughEvent(this, config, control));
        }

        return result;
    }

    private int getEveryAdGroupIdCount(String dimension, String adGroupId) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            return NumberUtils.toInt(jedis.get(strAdGroupCount(dimension, adGroupId).generateKey()));
        }
    }

    private List<FeedbackRequest> getConversionsFromDB(PositionRtb positionRtbConfig, PlusControlConfig config, Integer remain, List<String> adGroupId) {
        List<FeedbackRequest> result = Lists.newArrayList();
        int currentPage = 1;
        Integer pageSize = remain;
        String positionId = config.getPositionId();
        try {
            do {
                QueryParam queryParam = QueryParam.builder()
                        .positionId(positionId)
                        .startTime(config.getStartTime())
                        .endTime(config.getEndTime())
                        .action(getAction(config.getActionType()))
                        .adGroupId(adGroupId)
                        .build();
                PageHelper.startPage(currentPage, pageSize);
                List<SrcAdUserAccessLog> list = srcAdUserAccessLogMapper.list(queryParam);
                log.info("DB data [current {} size {}] {}",currentPage, pageSize, JSON.toJSONString(list));
                if (CollectionUtils.isEmpty(list)) {
                    break;
                }

                for (SrcAdUserAccessLog srcAdUserAccessLog : list) {
                    FeedbackRequest of = of(srcAdUserAccessLog, positionRtbConfig, config);
                    if (Objects.nonNull(of)
                            && !existClickId(positionId, of.getClickId())
                            && remain > result.size()
                            && adGroupId.contains(of.getAdGroupId())) {
                        result.add(of);
                    }
                }
                currentPage++;
                pageSize++;
            } while (remain > result.size());

        } catch (Exception e) {
            log.error("getConversionsFromDB occur error", e);
        }
        return result;
    }

    public FeedbackRequest of(SrcAdUserAccessLog log, PositionRtb positionRtbConfig, PlusControlConfig plusControlConfig) {
        String rtbExt = log.getRtbExt();

        String clickId = null;
        if (StringUtils.isNotBlank(rtbExt)) {
            JSONObject rtbExtJsonObject = JsonHelper.convert(rtbExt, JSONObject.class);
            clickId = rtbExtJsonObject.getString("click_id");
        }

        KeyValueMap keyValueMap = new KeyValueMap();
        keyValueMap.put(Inads.Param.OAID, log.getOaid());
        keyValueMap.put(Inads.Param.OAID_MD5, log.getOaidMd5());
        keyValueMap.put(Inads.Param.IMEI, log.getImei());
        keyValueMap.put(Inads.Param.IMEI_MD5, log.getImeiMd5());
        keyValueMap.put(Inads.Param.IDFA, log.getIdfa());
        keyValueMap.put(Inads.Param.IDFA_MD5, log.getIdfaMd5());
        keyValueMap.put(Inads.Param.ANDROIDID, log.getAnid());
        keyValueMap.put(Inads.Param.ANDROIDID_MD5, log.getAnidMd5());
        keyValueMap.put(Inads.Param.BXMID, log.getBxmId());
        keyValueMap.put(Inads.Param.IP, log.getIp());
        keyValueMap.put(Inads.Param.UA, log.getUas());
        keyValueMap.put(Inads.Param.UID, log.getUid());
        keyValueMap.put(Inads.Param.TIME, log.getTimeStamp());
        keyValueMap.put(Inads.Param.ADID, log.getPreId());
        keyValueMap.put(Inads.Param.TAGID, positionRtbConfig.getPositionId());

        return FeedbackRequest
                .builder()
                .config(positionRtbConfig)
                .conversionLevel(FeedbackRequest.SHALLOW_CONVERSION_LEVEL)
                .conversionType("0")
                .smartConvType(SmartConvType.NONE)
                .referrer(log.getRefer())
                .adGroupId(log.getAdGroupId())
                .clickId(clickId)
                .eventType(positionRtbConfig.getTargetOneRtb())
                .keyValueMap(keyValueMap)
                .actionType(plusControlConfig.getActionType())
                .build();
    }

    private boolean existClickId(String positionId, String clickId) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            Boolean existClickId = jedis.exists(RedisKeyGenerator.strFeedbackClickId(positionId, clickId).generateKey());
            if (existClickId) {
                log.info("Plus Control the clickId has been feedback. {}", clickId);
            }
            return existClickId;
        }
    }

    private Integer getAction(ActionType actionType) {
        return ActionType.TICKET_CLICK.equals(actionType) || ActionType.TICKET_CONVERSION.equals(actionType) ? 2 : 1;
    }

    private static KeyGenerator strAdGroupCount(String dimension, String adGroupId) {
        return () -> KeyBuilder.build("rtb_control", "plus", "count", dimension, adGroupId);
    }

}
