package com.bxm.adscounter.rtb.common.control.rate;

import com.bxm.adscounter.rtb.common.mapper.SrcAdUserAccessLogMapper;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.zk.ZkClientHolder;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.*;
import java.util.List;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 基于 Redis 实现的速率控制器
 *
 * @author allen
 * @date 2022-07-13
 * @since 1.0
 */
@Slf4j
@Deprecated
public class RedisRateControlImpl implements RateControl {

    private static final String SPLIT = "|";

    private final RateControlConfig config;
    private final ScheduledThreadPoolExecutor control = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("control"));
    private final JedisPool jedisPool;
    private final MeterRegistry registry;
    private final EventPark eventPark;

    private final RedisRateControlScheduler scheduler;
    private final LeaderLatch leaderLatch;

    public RedisRateControlImpl(RateControlConfig config, JedisPool jedisPool, MeterRegistry registry, EventPark eventPark, ZkClientHolder zkClientHolder, SrcAdUserAccessLogMapper srcAdUserAccessLogMapper) {
        this.eventPark = eventPark;
        Preconditions.checkNotNull(config);
        Preconditions.checkArgument(StringUtils.isNotBlank(config.getDimension()), "dimension cannot be null");
        Preconditions.checkNotNull(config.getConsumer(), "consumer cannot be null");
        long count = config.getCount();
        Preconditions.checkArgument(count > 0 && count < 10000, "count must > 0 and < 10000");

        this.config = config;
        this.jedisPool = jedisPool;
        this.registry = registry;
        this.scheduler = new RedisRateControlScheduler(this, jedisPool, eventPark, srcAdUserAccessLogMapper, registry);
        this.leaderLatch = new LeaderLatch(zkClientHolder.get(), "/adscounter/rtb/control/rate/" + config.getDimension());
    }

    @Override
    public void start() {
        this.leaderLatch.addListener(new LeaderLatchListener() {
            @Override
            public void isLeader() {
                long millis = Duration.between(LocalTime.now(), config.getEnd()).toMillis();
                long delay = millis / (config.getCount() - getCount());
                if (log.isInfoEnabled()) {
                    log.info("[{}] Rate Control Start, Delay time is {} ms", config.getDimension(), delay);
                }
                control.scheduleWithFixedDelay(scheduler, 0, delay, TimeUnit.MILLISECONDS);
            }

            @Override
            public void notLeader() {
            }
        });
        try {
            this.leaderLatch.start();
        } catch (Exception e) {
            log.error("start: ", e);
        }
    }

    @Override
    public void shutdown() {
        try {
            leaderLatch.close();
        } catch (Exception e) {
            log.error("shutdown: ", e);
        }
        control.shutdownNow();
    }

    @Override
    public void count() {
        this.addCount();
        this.registry.counter("rate.control.count", tags()).increment();
    }

    @Override
    public long getCount() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            return NumberUtils.toLong(jedis.get(string(config.getDimension()).generateKey()));
        }
    }

    @Override
    public void push(String id) {
        this.lpush(id);
        this.registry.counter("rate.control.list", tags()).increment();
    }

    @Override
    public void refresh() {
        this.control.remove(this.scheduler);
        this.start();
    }

    @Override
    public void delete() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String dimension = config.getDimension();
            String count = string(dimension).generateKey();
            String list = list(dimension).generateKey();
            jedis.del(count, list);
        }
    }

    @Override
    public RateControlConfig getConfig() {
        return this.config;
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }

    private void addCount() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = string(config.getDimension()).generateKey();
            Long rs = jedis.incr(key);
            if (log.isDebugEnabled()) {
                log.debug("{} after decrBy {}", key, rs);
            }
        }
    }

    private void lpush(String id) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            jedis.lpush(list(config.getDimension()).generateKey(), createLargeId(id));
        }
    }

    private List<Tag> tags() {
        return Lists.newArrayList(Tag.of("dim", config.getDimension()));
    }

    static String createLargeId(String id) {
        return System.currentTimeMillis() + SPLIT + id;
    }

    static KeyGenerator string(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "rate", "count", dimension);
    }

    static KeyGenerator list(String dimension) {
        return () -> KeyBuilder.build("rtb_control", "rate", "list", dimension);
    }

    static String[] splitLargeId(String id) {
        int i = id.indexOf(SPLIT);
        if (i == -1) {
            throw new IllegalStateException(String.format("%s is illegal value", id));
        }
        String t = id.substring(0, i);
        String d = id.substring(i + 1);
        return new String[] { t, d };
    }

    static LocalDateTime ofTimeMillis(long millis) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.systemDefault());
    }
}
