package com.bxm.adscounter.rtb.common.control.ratio;

import com.bxm.adscounter.rtb.common.ClickTracker;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.RtbIntegrationFactory;
import com.bxm.adscounter.rtb.common.control.RatioControlRtbIntegration;
import com.bxm.adscounter.rtb.common.service.PositionRtbService;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * 智能回传通用对接，不需要为每个 RTB 实例去单独实现，只要实现符合要求的 {@link ClickTracker} 即可。
 *
 * @author allen
 * @date 2022-10-18
 * @since 1.0
 */
public abstract class AbstractSmartConversionIntegration {

    private final PositionRtbService positionRtbService;
    private final RtbIntegrationFactory rtbIntegrationFactory;
    private final RatioControlRtbIntegration ratioControlRtbIntegration;

    protected AbstractSmartConversionIntegration(PositionRtbService positionRtbService, RtbIntegrationFactory rtbIntegrationFactory, RatioControlRtbIntegration ratioControlRtbIntegration) {
        this.positionRtbService = positionRtbService;
        this.rtbIntegrationFactory = rtbIntegrationFactory;
        this.ratioControlRtbIntegration = ratioControlRtbIntegration;
    }

    /**
     * 包含在内的 RTB 平台才会执行
     * @return 包含的集合
     */
    protected abstract Set<Rtb> includes();

    /**
     * 必须排除在外的 RTB 平台才会执行
     * @return 排除的集合
     */
    protected abstract Set<Rtb> excludes();

    protected void doTicketClick(KeyValueMap clickLog) {
        String tagid = clickLog.getFirst(Inads.Param.TAGID);
        PositionRtb config = positionRtbService.get(tagid);
        if (Objects.isNull(config)) {
            return;
        }

        Integer sourceType = config.getSourceType();
        Rtb rtb = Rtb.of(sourceType);
        if (Objects.isNull(rtb)) {
            return;
        }

        Set<Rtb> includes = includes();
        if (CollectionUtils.isNotEmpty(includes)) {
            Optional<Rtb> first = includes.stream().filter(e -> e == rtb).findFirst();
            if (!first.isPresent()) {
                return;
            }
        }
        Set<Rtb> excludes = excludes();
        if (CollectionUtils.isNotEmpty(excludes)) {
            Optional<Rtb> first = excludes.stream().filter(e -> e == rtb).findFirst();
            if (first.isPresent()) {
                return;
            }
        }

        RtbIntegration rtbIntegration = rtbIntegrationFactory.get(rtb);
        if (!(rtbIntegration instanceof ClickTracker)) {
            return;
        }

        ClickTracker clickTracker = (ClickTracker) rtbIntegration;
        String clickId = clickTracker.getClickIdOnInadsAdClickLog(clickLog);
        if (StringUtils.isBlank(clickId)) {
            return;
        }
        KeyValueMap clickTrackerEventLog = clickTracker.getClickTracker(clickId);
        if (Objects.isNull(clickTrackerEventLog)) {
            return;
        }
        String adGroupId = clickTrackerEventLog.getFirst(Common.Param.AD_GROUP_ID);
        String app = clickTracker.getApp(clickLog);

        ratioControlRtbIntegration.onTicketClick(rtbIntegration, clickTrackerEventLog, clickLog, adGroupId, app);
    }
}
