package com.bxm.adscounter.rtb.common.control.ratio;

import com.alibaba.fastjson.JSONArray;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.control.ControlUtils;
import com.bxm.adscounter.rtb.common.control.plus.PlusQueueService;
import com.bxm.adscounter.rtb.common.control.cpa.CpaControl;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioDeductionEvent;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioFeedbackEvent;
import com.bxm.adscounter.rtb.common.control.ratio.event.RatioFeedbackExceptionEvent;
import com.bxm.adscounter.rtb.common.control.ratio.listener.ListenerKey;
import com.bxm.adscounter.rtb.common.data.AdGroupData;
import com.bxm.adscounter.rtb.common.event.RtbDeductionEvent;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.*;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 基于 Redis 实现的比例控制器实现
 *
 * @author allen
 * @date 2022-07-11
 * @since 1.0
 */
@Slf4j
public class RedisRatioControlImpl implements RatioControl {

    static final int ONE_DAY_SEC = TypeHelper.castToInt(Duration.ofDays(1).getSeconds());

    private static final String SPLIT = "|";

    private RatioControlBus bus;
    private RatioControlConfig config;
    private final ScheduledThreadPoolExecutor computeScheduler;
    private final ScheduledThreadPoolExecutor dataFetchScheduler;
    private final JedisPool jedisPool;
    private final MeterRegistry registry;
    private final EventPark eventPark;

    private final RedisRatioControlScheduler scheduler;
    private final DataFetchingScheduler dataFetchingScheduler;
    private final LeaderLatch leaderLatch;
    private final CpaControl cpaControl;
    private final PlusQueueService plusQueueService;

    private final AtomicBoolean leader = new AtomicBoolean(false);
    private final AtomicBoolean computerIsRunning = new AtomicBoolean(false);
    private final AtomicBoolean dataFetchIsRunning = new AtomicBoolean(false);

    private final String hostAddress;

    public RedisRatioControlImpl(RatioControlBus bus, RatioControlConfig config, CpaControl cpaControl, PlusQueueService plusQueueService) {
        this.plusQueueService = plusQueueService;
        Preconditions.checkNotNull(config);
        Preconditions.checkNotNull(bus.getInstance());
        Preconditions.checkNotNull(bus.getJedisPool());
        Preconditions.checkNotNull(bus.getRegistry());
        Preconditions.checkNotNull(bus.getEventPark());
        Preconditions.checkNotNull(bus.getZkClientHolder());
        Preconditions.checkNotNull(bus.getFactory());
        Preconditions.checkArgument(StringUtils.isNotBlank(config.getDimension()), "dimension cannot be null");
        Preconditions.checkArgument(config.getExpireInHours() > 0, "expire hours must > 0");

        this.bus = bus;
        this.eventPark = bus.getEventPark();
        this.registry = bus.getRegistry();
        this.jedisPool = bus.getJedisPool();
        this.config = config;
        this.scheduler = new RedisRatioControlScheduler(this);
        this.dataFetchingScheduler = new DataFetchingScheduler(this);
        this.leaderLatch = new LeaderLatch(bus.getZkClientHolder().get(), "/adscounter/rtb/control/ratio/" + config.getDimension());
        this.hostAddress = getHost();
        this.computeScheduler = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("cs"));
        this.dataFetchScheduler = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("dfs"));
        this.cpaControl = cpaControl;
    }

    @Override
    public void start() {
        this.leaderLatch.addListener(new LeaderLatchListener() {
            @Override
            public void isLeader() {
                leader.compareAndSet(false, true);
                startScheduler(true);
            }

            @Override
            public void notLeader() {
                leader.compareAndSet(true, false);
            }
        });
        try {
            this.leaderLatch.start();
        } catch (Exception e) {
            log.error("start: ", e);
        }
    }

    /**
     * 开始执行调度任务。允许重复执行，最多只会创建一个任务。
     * 只有当前实例被选举为 Leader 才能执行。
     */
    private void startScheduler(boolean showLeaderLog) {
        if (!leader.get()) {
            if (showLeaderLog && log.isInfoEnabled()) {
                log.info("[{}] {} is not leader.", config.getDimension(), hostAddress);
            }
        } else {
            if (showLeaderLog && log.isInfoEnabled()) {
                log.info("[{}] {} is leader!", config.getDimension(), hostAddress);
            }

            // 这里还需要考虑开启调度器之后，如果下一个周期需要关闭，该如何停止任务释放资源。
            // 可以使用：scheduler.remove(RunnableScheduledFuture<?>)

            if (isAllowScheduleComputer()) {
                if (computerIsRunning.compareAndSet(false, true)) {
                    int intervalInSec = config.getIntervalInSec();
                    if (log.isInfoEnabled()) {
                        log.info("[{}] {} Computer Start, Delay time is {} sec", config.getDimension(), hostAddress, intervalInSec);
                    }
                    computeScheduler.scheduleWithFixedDelay(scheduler, 0, intervalInSec, TimeUnit.SECONDS);
                }
            }

            if (isAllowScheduleDataFetcher()) {
                if (dataFetchIsRunning.compareAndSet(false, true)) {
                    int dataFetchInMinute = config.getDataFetchInMinute();
                    if (log.isInfoEnabled()) {
                        log.info("[{}] {} DataFetcher Start, Delay time is {} min", config.getDimension(), hostAddress, dataFetchInMinute);
                    }
                    dataFetchScheduler.scheduleWithFixedDelay(dataFetchingScheduler, 0, dataFetchInMinute, TimeUnit.MINUTES);
                }
            }
        }
    }

    /**
     * @return 是否允许执行计算器
     */
    private boolean isAllowScheduleComputer() {
        return config.isEnableCvrControl() || config.isEnableCostControl();
    }

    /**
     * @return 是否允许执行数据获取器
     */
    private boolean isAllowScheduleDataFetcher() {
        return config.isEnableCostControl();
    }

    @Override
    public void shutdown() {
        try {
            leaderLatch.close();
        } catch (Exception e) {
            log.error("shutdown: ", e);
        }
        computeScheduler.shutdownNow();
        dataFetchScheduler.shutdownNow();
    }

    @Override
    public void onClick(String clickId, String app) {
        this.incrClickTotalCount(clickId);
        this.incrClickTotalCount(clickId, app);
        this.registry.counter("ratio.control.denominator", tags()).increment();
    }

    @Override
    public void onAdClick(ClickLog clickLog) {
        this.addClickQueue(clickLog);
        // 券点击排序
        this.addTicketClick2SortApp(clickLog.getApp());
        this.registry.counter("ratio.control.adclick", tags()).increment();
    }

    @Override
    public void onConversion() {
        this.incrConversionTotalCount();
        this.registry.counter("ratio.control.numerator", tags()).increment();
    }

    @Override
    public void pushConversion(FeedbackRequest request, String app) {
        String data = JsonHelper.convert(request);

        this.incrReceiverTotalCount();

        // 转化数排序
        double currentConvCount = this.addConversion2SortApp(app);
        // CVR排序
        this.setCvr2SortApp(app, currentConvCount / this.getClickTotalCount(app));

        if (config.isEnableDeductionControl()) {
            if (isDeduction(request)) {
                this.addConversion(config.getDimension(), data);
                this.addAppConversion(config.getDimension(), app, data);
                cpaControl.pushConversion(request);
                plusQueueService.pushConversion(request);
            } else {
                // 回传
                String adGroupId = request.getAdGroupId();
                String appid = request.getAppid();
                try {
                    FeedbackResponse response = bus.getInstance().doFeedback(request, 1);
                    if (response.isSuccess()) {
                        this.onConversion();
                        eventPark.post(new RatioFeedbackEvent(this, this, adGroupId, appid, null));
                    } else {
                        throw new RtbIntegrationException(null);
                    }
                } catch (RtbIntegrationException e) {
                    eventPark.post(new RatioFeedbackExceptionEvent(this, this, adGroupId, appid, null));
                } catch (Exception e) {
                    log.error("occur exception | accept: ", e);
                }
            }
        } else {
            this.addConversion(config.getDimension(), data);
            this.addAppConversion(config.getDimension(), app, data);
        }
        this.registry.counter("ratio.control.numerator.list", tags()).increment();
    }

    @Override
    public void onX() {
        this.incrX(1);
    }

    @Override
    public void cleanX() {
        this.delX();
    }

    /**
     * 当前转化请求是否被扣下
     *
     * @param request 请求
     * @return 如果被扣下，则返回 true
     */
    private boolean isDeduction(FeedbackRequest request) {
        int level = request.getConversionLevel();
        double stepValue = 1d;
        double zeroRatio = 0d;
        double ratio = request.isDeepConversion() ? config.getDeductionDeepRatio() : config.getDeductionShallowRatio();
        if (!this.isInitializedCountRate(level) && ratio > zeroRatio) {
            this.incrCountRateBy(level, stepValue);
        }

        this.registry.counter("deduction.control", tags2(level)).increment();

        double after = this.incrCountRateBy(level, ratio);
        boolean feedbackIfNecessary = after >= stepValue;
        if (feedbackIfNecessary) {
            // 回传
            this.incrCountRateBy(level, - stepValue);
            this.registry.counter("deduction.control.accept", tags2(level)).increment();
            return false;
        } else {
            // 扣量
            Rtb rtb = bus.getInstance().rtb();
            this.eventPark.post(new RtbDeductionEvent(this, rtb, request, null));
            this.eventPark.post(new RatioDeductionEvent(this, this, request.getAdGroupId(), request.getAppid()));
            this.registry.counter("deduction.control.reject", tags2(level)).increment();
            return true;
        }
    }

    @Override
    public void delete() {
    }

    @Override
    public Stat getStat() {
        long conversions = 0, clicks = 0, x = 0, plus = 0, receivers = 0;
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            // 分子
            conversions = NumberUtils.toLong(jedis.get(stringConversionTotal(config.getDimension()).generateKey()), 0);
            // 去重后的分母
            clicks = NumberUtils.toLong(jedis.get(stringClickTotal(config.getDimension()).generateKey()), 0);
            // COST 当前周期已经回传的数量
            x = NumberUtils.toLong(jedis.get(stringX(bus.getInstance().rtb(), config.getAdGroupId()).generateKey()), 0);
            // 非转化补量数
            plus = NumberUtils.toLong(jedis.hget(ListenerKey.hashData(config.getDimension()).generateKey(), ListenerKey.Field.PLUS), 0);
            // 实际转化数
            receivers = NumberUtils.toLong(jedis.get(stringReceiverTotal(config.getDimension()).generateKey()), 0);
        }

        double v = clicks == 0 ? 0 : BigDecimal.valueOf(conversions)
                .divide(BigDecimal.valueOf(clicks), 3, RoundingMode.HALF_UP)
                .doubleValue();

        if (log.isDebugEnabled()) {
            log.debug("[{}] Current ratio: {}/{} = {}", config.getDimension(), conversions, clicks, v);
        }
        return new Stat()
                .setConversions(conversions)
                .setClicks(clicks)
                .setCvr(v)
                .setX(x)
                .setPlus(plus)
                .setReceivers(receivers);
    }

    @Override
    public Set<String> getTopN(int start, int stop) {
        if (config.getPlusStrategy() == RatioControlConfig.PlusStrategy.CVR) {
            return orderByCvr(start, stop);
        } else {
            return orderByConversion(start, stop);
        }
    }

    @Override
    public void pushData(List<AdGroupData> dataList) {
        if (CollectionUtils.isEmpty(dataList)) {
            return;
        }
        this.pushAdGroupData(dataList);
    }

    @Override
    public List<AdGroupData> getDataList() {
        return this.getAdGroupData();
    }

    @Override
    public RatioControlBus getBus() {
        return this.bus;
    }

    @Override
    public void refreshConfig(RatioControlConfig config) {
        this.config = config;
        this.startScheduler(false);
    }

    @Override
    public RatioControlConfig getConfig() {
        return this.config;
    }

    private List<Tag> tags() {
        return Lists.newArrayList(Tag.of("dim", config.getDimension()));
    }

    private List<Tag> tags2(int level) {
        return Lists.newArrayList(
                Tag.of("dim", config.getDimension() + ControlUtils.KEY_SPLIT_CHAR + level)
        );
    }

    private JedisPool getJedisPool() {
        return this.jedisPool;
    }

    // ----------------- Deduction redis operations ------------------ //

    private boolean isInitializedCountRate(int level) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringRateCountInitialized(config.getDimension(), level).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
            return Optional.ofNullable(rs).orElse(1L) > 1;
        }
    }

    private double incrCountRateBy(int level, double incrementValue) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringCountRate(config.getDimension(), level).generateKey();
            double rs = jedis.incrByFloat(key, incrementValue);
            jedis.expire(key, ONE_DAY_SEC);
            return rs;
        }
    }

    private void incrClickTotalCount(String clickId) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringClickTotal(config.getDimension()).generateKey();
            jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
        }
    }

    private void incrClickTotalCount(String clickId, String app) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringClickTotal(config.getDimension(), app).generateKey();
            jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
        }
    }

    private long getClickTotalCount(String app) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringClickTotal(config.getDimension(), app).generateKey();
            return NumberUtils.toLong(jedis.get(key), 0);
        }
    }

    private void addClickQueue(ClickLog clickLog) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = listClickQueue(config.getDuration(), config.getDimension(), clickLog.getApp()).generateKey();
            jedis.lpush(key, JsonHelper.convert(clickLog));
            jedis.expire(key, getRemainInSeconds());
        }
    }

    private void incrReceiverTotalCount() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringReceiverTotal(config.getDimension()).generateKey();
            jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
        }
    }

    private void incrConversionTotalCount() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringConversionTotal(config.getDimension()).generateKey();
            jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
        }
    }

    private double addConversion2SortApp(String app) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = sortsetConversion(config.getDuration(), config.getDimension()).generateKey();
            Double rs = jedis.zincrby(key, 1, app);
            jedis.expire(key, ONE_DAY_SEC);
            return rs;
        }
    }

    private void addTicketClick2SortApp(String app) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = sortsetTicketClick(config.getDuration(), config.getDimension()).generateKey();
            jedis.zincrby(key, 1, app);
            jedis.expire(key, ONE_DAY_SEC);
        }
    }

    private void setCvr2SortApp(String app, double cvr) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = sortsetCvr(config.getDuration(), config.getDimension()).generateKey();
            jedis.zadd(key, cvr, app);
            jedis.expire(key, ONE_DAY_SEC);
        }
    }

    private void addConversion(String dimension, String data) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = listConversionQueue(config.getDuration(), dimension).generateKey();
            jedis.lpush(key, createLargeId(data));
            jedis.expire(key, config.getExpireInSeconds());
        }
    }

    private void addAppConversion(String dimension, String app, String data) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = listAppConversionQueue(config.getDuration(), dimension, app).generateKey();
            jedis.lpush(key, createLargeId(data));
            jedis.expire(key, config.getExpireInSeconds());
        }
    }

    private Set<String> orderByConversion(int start, int stop) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = sortsetConversion(config.getDuration(), config.getDimension()).generateKey();
            return jedis.zrevrange(key, start, stop);
        }
    }

    private Set<String> orderByTicketClick(int start, int stop) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = sortsetTicketClick(config.getDuration(), config.getDimension()).generateKey();
            return jedis.zrevrange(key, start, stop);
        }
    }

    private Set<String> orderByCvr(int start, int stop) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = sortsetCvr(config.getDuration(), config.getDimension()).generateKey();
            return jedis.zrevrange(key, start, stop);
        }
    }

    private void pushAdGroupData(List<AdGroupData> dataList) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringRtbData(bus.getInstance().rtb(), config.getAdGroupId()).generateKey();
            jedis.set(key, JsonHelper.convert(dataList));
            jedis.expire(key, config.getDataExpireInSeconds());
        }
    }

    private List<AdGroupData> getAdGroupData() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringRtbData(bus.getInstance().rtb(), config.getAdGroupId()).generateKey();
            String rs = jedis.get(key);
            return JSONArray.parseArray(rs, AdGroupData.class);
        }
    }

    private void incrX(long incrementValue) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringX(bus.getInstance().rtb(), config.getAdGroupId()).generateKey();
            Long rs = jedis.incrBy(key, incrementValue);
            jedis.expire(key, config.getDataExpireInSeconds());
            if (log.isDebugEnabled()) {
                log.debug("X - {} - {}", key, rs);
            }
        }
    }

    private void delX() {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringX(bus.getInstance().rtb(), config.getAdGroupId()).generateKey();
            jedis.del(key);
            if (log.isDebugEnabled()) {
                log.debug("Delete X - {}", key);
            }
        }
    }

    // ----------- common static functions ----------- //

    private static String createLargeId(String data) {
        return System.currentTimeMillis() + SPLIT + data;
    }

    // ----------- Deduction control Keys ------------ //

    private static KeyGenerator stringRateCountInitialized(String dimension, int level) {
        return () -> KeyBuilder.build("rtb", "conv", "DED_INIT", getDate(), dimension, level);
    }

    private static KeyGenerator stringCountRate(String dimension, int level) {
        return () -> KeyBuilder.build("rtb", "conv", "DED_RATE", getDate(), dimension, level);
    }

    // ----------- Cvr control Keys ------------ //

    /**
     * 接收广告主转化总数
     *
     * @param dimension 维度
     * @return 键
     */
    private static KeyGenerator stringReceiverTotal(String dimension) {
        return () -> KeyBuilder.build("rtb", "conv", "RECEIVER", getDate(), dimension);
    }

    /**
     * 总点击数
     *
     * @param dimension 维度
     * @return 键
     */
    private static KeyGenerator stringClickTotal(String dimension) {
        return () -> KeyBuilder.build("rtb", "conv", "CLICK", getDate(), dimension);
    }

    /**
     * 媒体 点击数
     *
     * @param dimension 维度
     * @return 键
     */
    private static KeyGenerator stringClickTotal(String dimension, String app) {
        return () -> KeyBuilder.build("rtb", "conv", "APP_CLICK", getDate(), dimension, app);
    }

    /**
     * 总转化数
     *
     * @param dimension 维度
     * @return 键
     */
    private static KeyGenerator stringConversionTotal(String dimension) {
        return () -> KeyBuilder.build("rtb", "conv", "CONVERSION", getDate(), dimension);
    }

    /**
     * 转化数
     *
     * Member: app
     *
     * @param dimension 维度
     * @return 键
     */
    private static KeyGenerator sortsetConversion(RatioControlConfig.Duration2 duration2, String dimension) {
        String split = duration2 == RatioControlConfig.Duration2.HOUR ? getDateHour() : getDate();
        return () -> KeyBuilder.build("rtb", "conv", "APP_CONV", split, dimension);
    }

    /**
     * 券点击数
     *
     * Member: app
     *
     * @param dimension 维度
     * @return 键
     */
    private static KeyGenerator sortsetTicketClick(RatioControlConfig.Duration2 duration2, String dimension) {
        String split = duration2 == RatioControlConfig.Duration2.HOUR ? getDateHour() : getDate();
        return () -> KeyBuilder.build("rtb", "conv", "APP_TICKET_CLICK", split, dimension);
    }

    /**
     * CVR
     *
     * Member: app
     *
     * @param dimension 维度
     * @return 键
     */
    private static KeyGenerator sortsetCvr(RatioControlConfig.Duration2 duration2, String dimension) {
        String split = duration2 == RatioControlConfig.Duration2.HOUR ? getDateHour() : getDate();
        return () -> KeyBuilder.build("rtb", "conv", "APP_CONV_CVR", split, dimension);
    }

    /**
     * 点击队列
     *
     * Value: ClickLog
     *
     * @param dimension 维度
     * @param app APP
     * @return 键
     */
    static KeyGenerator listClickQueue(RatioControlConfig.Duration2 duration2, String dimension, String app) {
        String split = duration2 == RatioControlConfig.Duration2.HOUR ? getDateHour() : getDate();
        return () -> KeyBuilder.build("rtb", "conv", "CLICK_QUEUE", split, dimension, app);
    }

    /**
     * 转化队列
     *
     * @param dimension 维度
     * @return 键
     */
    static KeyGenerator listConversionQueue(RatioControlConfig.Duration2 duration2, String dimension) {
        String split = duration2 == RatioControlConfig.Duration2.HOUR ? getDateHour() : getDate();
        return () -> KeyBuilder.build("rtb", "conv", "CONV_QUEUE", split, dimension);
    }

    /**
     * APP 维度转化队列
     *
     * @param dimension 维度
     * @param app APP
     * @return 键
     */
    static KeyGenerator listAppConversionQueue(RatioControlConfig.Duration2 duration2, String dimension, String app) {
        String split = duration2 == RatioControlConfig.Duration2.HOUR ? getDateHour() : getDate();
        return () -> KeyBuilder.build("rtb", "conv", "APP_CONV_QUEUE", split, dimension, app);
    }

    /**
     * RTB 数据
     *
     * @param rtb RTB
     * @param adGroupId ad_group_id
     * @return 键
     */
    static KeyGenerator stringRtbData(Rtb rtb, String adGroupId) {
        return () -> KeyBuilder.build("rtb", "conv", "RTB_DATA", rtb.getType(), adGroupId);
    }

    /**
     * 使用效果成本控制时，当前周期已经回传的数量
     *
     * @param rtb RTB
     * @param adGroupId ad_group_id
     * @return 键
     */
    static KeyGenerator stringX(Rtb rtb, String adGroupId) {
        return () -> KeyBuilder.build("rtb", "conv", "X", rtb.getType(), adGroupId);
    }

    static String[] splitLargeId(String id) {
        int i = id.indexOf(SPLIT);
        if (i == -1) {
            throw new IllegalStateException(String.format("%s is illegal value", id));
        }
        String t = id.substring(0, i);
        String d = id.substring(i + 1);
        return new String[] { t, d };
    }

    static KeyGenerator listTicketConversionQueue(String dimension) {
        return () -> KeyBuilder.build("rtb", "conv", "TICKET_CONV_QUEUE", getDate(), dimension);
    }

    static LocalDateTime ofTimeMillis(long millis) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneId.systemDefault());
    }

    private static String getDate() {
        return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
    }

    private static String getDateHour() {
        return LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHH"));
    }

    private static int getRemainInSeconds() {
        return (int) DateHelper.getRemainSecondsOfToday() + RandomUtils.nextInt(600);
    }

    private static String getHost() {
        try {
            return InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            return "unknown";
        }
    }

    public EventPark getEventPark() {
        return eventPark;
    }
}
