package com.bxm.adscounter.rtb.common.control.ticket;

import com.bxm.adscounter.rtb.common.ClickTracker;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegration;
import com.bxm.adscounter.rtb.common.control.ControlUtils;
import com.bxm.adscounter.rtb.common.event.RtbDeductionEvent;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.utils.PositionRTBUtils;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Optional;

/**
 * @author tangxiao
 * @date 2023/2/28
 * @since 1.0
 */
@Slf4j
public class RedisTicketRatioControlImpl implements TicketRatioControl{


    private static final String SPLIT = "|";
    private static final Integer ONE_DAY_SEC = (int) Duration.ofHours(24).getSeconds();

    private final JedisPool jedisPool;
    private final EventPark eventPark;
    private MeterRegistry registry;

    public RedisTicketRatioControlImpl(JedisPool jedisPool, EventPark eventPark) {
        this.jedisPool = jedisPool;
        this.eventPark = eventPark;
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
    }

    @Override
    public boolean onControl(FeedbackRequest request, RtbIntegration rtbIntegration) {

        // 互动累加模式
        PositionRtb config = request.getConfig();
        if (config.isFeedbackModeSummaryCpa()) {
            return false;
        }

        if (rtbIntegration instanceof ClickTracker) {
            boolean isDeduction = isDeduction(request);
            if (isDeduction) {
                // 放队列了 维度：广告位+广告组id
                pushConversion(request);
                return true;
            }
        }
        return false;
    }

    public boolean isDeduction(FeedbackRequest request) {
        PositionRtb config = request.getConfig();
        KeyValueMap keyValueMap = request.getKeyValueMap();
        String ticketId = keyValueMap.getFirst(Inads.Param.ADID);
        if (StringUtils.isBlank(ticketId)) {
            return false;
        }

        PositionRtb.TargetBXM targetBXM = PositionRTBUtils.convert(ticketId, config);
        Double ratio = Optional.ofNullable(targetBXM.getRatio()).orElse(1d);
        if (ratio == 1d) {
            return false;
        }

        String positionId = config.getPositionId();
        String adGroupId = request.getAdGroupId();
        String dimension = ControlUtils.createKey(positionId, adGroupId, ticketId);
        Integer sourceType = config.getSourceType();
        Rtb rtb = Rtb.of(sourceType);
        return isDeduction0(ratio, dimension, request, rtb);
    }

    private boolean isDeduction0(Double ratio, String dimension, FeedbackRequest request, Rtb rtb) {
        double stepValue = 1d;
        double zeroRatio = 0d;
        if (!this.isInitializedCountRate(dimension) && ratio > zeroRatio) {
            this.incrCountRateBy(stepValue, dimension);
        }

        this.registry.counter("deduction.ticket.control", tags(dimension)).increment();

        double after = this.incrCountRateBy(ratio, dimension);
        boolean feedbackIfNecessary = after >= stepValue;
        if (feedbackIfNecessary) {
            // 回传
            this.incrCountRateBy(-stepValue, dimension);
            this.registry.counter("deduction.ticket.control.accept", tags(dimension)).increment();
            return false;
        } else {
            // 扣量
            this.eventPark.post(new RtbDeductionEvent(this, rtb, request, RtbDeductionEvent.DEDUCTION_TICKET));
            this.registry.counter("deduction.ticket.control.reject", tags(dimension)).increment();
            return true;
        }
    }

    private void pushConversion(FeedbackRequest request) {
        // 这里的 dimension 字符串如果调整，需要同步修改下面这些地方：
        // com.bxm.adscounter.rtb.common.control.ratio.RedisRatioControlScheduler.getOnTicketConversionQueueOrderBy
        String dimension = ControlUtils.createKey(request.getConfig().getPositionId(), request.getAdGroupId());
        String data = JsonHelper.convert(request);
        addConversion(dimension, data);
    }

    private void addConversion(String dimension, String data) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = listConversionQueue(dimension).generateKey();
            jedis.lpush(key, createLargeId(data));
            jedis.expire(key, getExpireInSeconds());
        }
    }

    private boolean isInitializedCountRate(String dimension) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringRateCountInitialized(dimension).generateKey();
            Long rs = jedis.incr(key);
            jedis.expire(key, ONE_DAY_SEC);
            return Optional.ofNullable(rs).orElse(1L) > 1;
        }
    }

    private double incrCountRateBy(double incrementValue, String dimension) {
        JedisPool jedisPool = getJedisPool();
        try (Jedis jedis = jedisPool.getResource()) {
            String key = stringCountRate(dimension).generateKey();
            double rs = jedis.incrByFloat(key, incrementValue);
            jedis.expire(key, ONE_DAY_SEC);
            return rs;
        }
    }

    private static KeyGenerator stringRateCountInitialized(String dimension) {
        return () -> KeyBuilder.build("rtb", "conv", "DED_TICKET_INIT", getDate(), dimension);
    }

    private static KeyGenerator stringCountRate(String dimension) {
        return () -> KeyBuilder.build("rtb", "conv", "DED_TICKET_RATE", getDate(), dimension);
    }

    static KeyGenerator listConversionQueue(String dimension) {
        return () -> KeyBuilder.build("rtb", "conv", "TICKET_CONV_QUEUE", getDate(), dimension);
    }

    public JedisPool getJedisPool() {
        return jedisPool;
    }

    private static String getDate() {
        return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
    }

    private static String createLargeId(String data) {
        return System.currentTimeMillis() + SPLIT + data;
    }

    public int getExpireInSeconds() {
        return (int) Duration.ofHours(24).getSeconds();
    }


    private List<Tag> tags(String dimension) {
        return Lists.newArrayList(Tag.of("dim", dimension));
    }
}
