package com.bxm.adscounter.rtb.common.impl.baicai;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adscounter.rtb.common.impl.mengran.MengranConfig;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Map;
import java.util.Objects;

/**
 * @author tangxiao
 * @date 2024-01-15
 * @since 1.0
 */
@Slf4j
public class BaicaiRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "bcx_cli";

    public BaicaiRtbIntegration(BaicaiConfig config) {
        super(config, createHttpClient());
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        if (Objects.isNull(keyValueMap)) {
            return null;
        }
        return keyValueMap.getFirst(Inads.Param.TAGID);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        KeyValueMap keyValueMap = request.getKeyValueMap();

        String clickId = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams().getFirst(CLICK_ID);
        if (StringUtils.isBlank(clickId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'bcx_cli' by referrer.");
        }

        Map<String, Object> params = Maps.newHashMap();
        params.put("url", referrer);
        params.put("ad_click", clickId);
        params.put("type", eventType);
        params.put("uid", keyValueMap.getFirst(Inads.Param.UID));

        url = OkHttpUtils.appendParams(url, params);
        return new HttpGet(url);
    }

    @Override
    public Rtb rtb() {
        return Rtb.BAICAI;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        Response response = JsonHelper.convert(body, Response.class);
        FeedbackResponse feedbackResponse =new FeedbackResponse();
        feedbackResponse.setSuccess(Objects.nonNull(response) && response.isSuccess());
        return feedbackResponse;
    }

    private static CloseableHttpClient createHttpClient() {
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(2000).setConnectionRequestTimeout(1000).setConnectTimeout(1000).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
    }

    @Data
    private static class Response {
        private String code;
        private String data;

        public boolean isSuccess() {
            return Objects.equals(code, "200");
        }
    }
}
