package com.bxm.adscounter.rtb.common.impl.baidu;

import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.rtb.common.DataFetchFailException;
import com.bxm.adscounter.rtb.common.DataFetcher;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.autoconfigure.RtbJedisProperties;
import com.bxm.adscounter.rtb.common.data.AdGroupData;
import com.bxm.adscounter.rtb.common.data.Parameter;
import com.bxm.adscounter.rtb.common.impl.AbstractHttpRtbIntegration;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.springframework.http.MediaType;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author tangxiao
 * @date 2023-03-20
 * @since 1.0
 */
@Slf4j
public class BaiduDataFetcher implements DataFetcher {

    private final BaiduConfig config;
    private final JedisPool jedisPool;
    private final HttpClient httpClient;

    private final static int UNIT_REPORT_TYPE = 2330652;

    public BaiduDataFetcher(BaiduConfig config, JedisPool jedisPool) {
        this.config = config;
        this.httpClient = AbstractHttpRtbIntegration.createHttpClient(config);
        this.jedisPool = jedisPool;
    }

    private static String getToday() {
        return getToday(0);
    }

    private static String getToday(long days) {
        return LocalDate.now().plusDays(days).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
    }

    private static String getHour(long hours) {
        return LocalDateTime.now().plusHours(hours).withMinute(0).format(DateTimeFormatter.ofPattern("HH"));
    }

    @Override
    public Rtb rtb() {
        return Rtb.Baidu;
    }

    @Override
    public List<AdGroupData> fetchCurrentHourData(Parameter parameter) throws DataFetchFailException {
        final String today = getToday();
        final String hour = getHour(0);
        return fetch0(parameter, today, hour);
    }

    @Override
    public List<AdGroupData> fetchPreviousHourData(Parameter parameter) throws DataFetchFailException {
        final String today = (LocalTime.now().getHour() == 0) ? getToday(-1) : getToday();
        final String hour = getHour(-1);
        return fetch0(parameter, today, hour);
    }

    private List<AdGroupData> fetch0(Parameter parameter, String date, String hour) throws DataFetchFailException {
        String tagId = parameter.getTagId();
        String advertiserId = parameter.getAdvertiserId();
        String advertiserName = parameter.getAdvertiserName();
        String adGroupId = parameter.getAdGroupId();

        String json = null;
        try {
            if (StringUtils.isBlank(advertiserId)) {
                throw new DataFetchFailException("'advertiserId' 不能为空，在广告位：" + tagId);
            }
            if (StringUtils.isBlank(advertiserName)) {
                throw new DataFetchFailException("'advertiserName' 不能为空，在广告位：" + tagId);
            }

            BaiduAk ak = getAk(Objects.toString(rtb().getType()), advertiserId);
            if (Objects.isNull(ak)) {
                throw new DataFetchFailException(String.format("没有找到 AK 密钥，在广告位：%s 和 advertiserId：%s", tagId, advertiserId));
            }

            String accessToken = ak.getAccessToken();

            JSONObject header = new JSONObject();
            header.put("userName", advertiserName);
            header.put("accessToken", accessToken);

            JSONObject body = new JSONObject();
            body.put("reportType", UNIT_REPORT_TYPE);
            body.put("userId", new Long[] {Long.valueOf(advertiserId) });
            body.put("timeUnit", "HOUR");
            body.put("startDate", date);
            body.put("endDate", date);
            body.put("columns", new String[] {
                    "date", "hour", "adGroupId", "click", "cost", "ocpcTransType",
                    "ocpcTargetTrans", "ocpcTargetTransCPC", "ocpcTargetTransRatio"
            });
            body.put("filters", buildDataFilter(adGroupId, hour));

            JSONObject req = new JSONObject();
            req.put("header", header);
            req.put("body", body);

            if (log.isInfoEnabled()) {
                log.info("baidu data request : {}", req.toJSONString());
            }

            HttpPost post = new HttpPost(config.getUrlReportData());
            post.setEntity(new StringEntity(req.toJSONString(), StandardCharsets.UTF_8));
            post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);

            HttpResponse hr = httpClient.execute(post);
            json = EntityUtils.toString(hr.getEntity());
            if (log.isInfoEnabled()) {
                log.info("baidu data response : {}", json);
            }
            BaiduUnitReportRes res = JSONObject.parseObject(json, BaiduUnitReportRes.class);
            if (!res.isSuccess()) {
                throw new DataFetchFailException(String.format("接口请求不成功！广告位：%s，advertiserId：%s。返回结果：%s", tagId, advertiserId, json));
            }
            return convert(res);
        } catch (IOException e) {
            throw new DataFetchFailException(String.format("接口网络异常（%s）！广告位：%s，advertiserId：%s。", e.getClass().getName(), tagId, advertiserId));
        } catch (JSONException e) {
            throw new DataFetchFailException(String.format("接口返回数据异常！广告位：%s，advertiserId：%s, advertiserName：%s。返回结果：%s", tagId, advertiserId, advertiserName, json));
        }
    }

    private BaiduAk getAk(String tagId, String advertiserId) {
        try (Jedis jedis = jedisPool.getResource()) {
            String v = jedis.hget(KeyBuilder.build("rtb", "conv", "AK", tagId), advertiserId);
            return Optional.ofNullable(v)
                    .filter(StringUtils::isNotBlank)
                    .map(s -> JsonHelper.convert(s, BaiduAk.class))
                    .orElse(null);
        }
    }

    private List<AdGroupData> convert(BaiduUnitReportRes res) {
        ResHeader header = res.getHeader();
        if (header == null) {
            return null;
        }
        Integer status = header.getStatus();
        if (!Objects.equals(status, 0)) {
            return null;
        }
        Body body = res.getBody();
        if (body == null) {
            return null;
        }
        List<RowData> rowData = body.getData();
        if (CollectionUtils.isEmpty(rowData)) {
            return null;
        }
        List<UnitRow> unitRows = rowData.iterator().next().getRows();
        if (CollectionUtils.isEmpty(unitRows)) {
            return null;
        }
        List<AdGroupData> ret = Lists.newArrayList();
        unitRows.forEach(e -> {
            AdGroupData adGroupData = new AdGroupData()
                    .setAdGroupId(e.getAdGroupId())
                    .setDate(e.getDate())
                    .setHour(e.getHour())
                    .setCharge(e.getCost())
                    .setTime(System.currentTimeMillis());

            adGroupData.setShallowConvCount(e.getOcpcTargetTrans());
            adGroupData.setShallowConvCost(e.getOcpcTargetTransCPC());
            adGroupData.setDeepConvCount(e.getOcpcTargetTrans());
            adGroupData.setDeepConvCost(e.getOcpcTargetTransCPC());

            adGroupData.setConvNumByImpression(e.getOcpcTargetTrans());

            ret.add(adGroupData);
        });

        log.info("ret {}", JSONObject.toJSONString(ret) );
        return ret;
    }

    public static class BaiduUnitReportRes extends Bd {}

    @Data
    public static class Bd{
        private ResHeader header;
        private Body body;

        public boolean isSuccess() {
            return header != null && header.getStatus() == 0;
        }
    }

    @Data
    public static class Body {
        private List<RowData> data;
    }

    @Data
    public static class RowData {
        private List<UnitRow> rows;
        private UnitRow summary;
        private Integer rowCount;
        private Integer totalRowCount;
    }

    @Data
    public static class UnitRow {
        private String date;
        private Integer hour;
        private String adGroupId;
        private Long click;
        private BigDecimal cost;
        private String ocpcTransType;
        private Long ocpcTargetTrans;
        private BigDecimal ocpcTargetTransCPC;
        private BigDecimal ocpcTargetTransRatio;
    }

    @Data
    public static class ResHeader {
        private int status = -1;
        private String desc;
        private Integer quota;
        private String failures;
        private Integer oprs;
        private String oprtime;
    }

    public static void main(String[] args) throws DataFetchFailException {
        RtbJedisProperties properties = new RtbJedisProperties();
        properties.setHost("47.97.189.222");
        properties.setPassword("redis_pwd123");
        properties.setDatabase(13);
        JedisPool jedisPool = new JedisPool(properties, properties.getHost(), properties.getPort(), properties.getTimeout(), properties.getPassword(), properties.getDatabase());

        BaiduDataFetcher baiduDataFetcher = new BaiduDataFetcher(new BaiduConfig(), jedisPool);
        Parameter parameter = Parameter.builder()
                .advertiserName("原生-微财234B22KA04069")
                .advertiserId("45003219")
                .adGroupId("7962468043")
                .tagId("d229c559c27245fc94dbcaa2e7d64b61-1")
                .build();

        baiduDataFetcher.fetch0(parameter, "2023-03-21", "16");
    }

    @Data
    @AllArgsConstructor
    @Builder
    private static class DataFilter {
        private String column;
        private String operator;
        private String[] values;
    }

    private static DataFilter[] buildDataFilter(String adGroupId, String hour) {
        DataFilter adGroupIdFilter = DataFilter.builder()
                .column("adGroupId")
                .operator("IN")
                .values(new String[]{adGroupId})
                .build();
        DataFilter hourFilter = DataFilter.builder()
                .column("hour")
                .operator("EQ")
                .values(new String[]{hour})
                .build();

        return new DataFilter[]{adGroupIdFilter, hourFilter};
    }

}
