package com.bxm.adscounter.rtb.common.impl.bxm;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * @author tangxiao
 * @date 2022-11-29
 * @since 1.0
 */
@Slf4j
public class BxmRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "click_id";
    public static final String AD_GROUP_ID = "ad_group_id";

    public BxmRtbIntegration(BxmConfig config) {
        super(config);
    }

    @Override
    public Rtb rtb() {
        return Rtb.BXM;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        return queryParams.getFirst(AD_GROUP_ID);
    }

    @Override
    protected boolean isReadBodyForHttpResponse() {
        return true;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String referrer = request.getReferrer();
        String eventType = request.getEventType();

        MultiValueMap<String, String> queryParams = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams();
        String clickid = queryParams.getFirst(getClickIdParameterNameOnReferrer());
        if (StringUtils.isBlank(clickid)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'click_id' by referrer.");
        }

        Map<String, Object> params = Maps.newHashMap();
        params.put("clickid", clickid);
        params.put("eventType", eventType);
        params.put("convTime", System.currentTimeMillis());

        String feedbackUrl = getFeedbackUrl();
        feedbackUrl = OkHttpUtils.appendParams(feedbackUrl, params);
        return new HttpGet(feedbackUrl);
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        BxmResponse response = JsonHelper.convert(body, BxmResponse.class);
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        if (Objects.nonNull(response) && response.isSuccess()) {
            feedbackResponse.setSuccess(true);
            feedbackResponse.setBody(response.getMessage());
        } else {
            feedbackResponse.setSuccess(false);
        }
        return feedbackResponse;
    }

    @Data
    private static class BxmResponse {
        private Integer code;
        private String message;

        public boolean isSuccess() {
            return Objects.equals(0, code);
        }
    }

}
