package com.bxm.adscounter.rtb.common.impl.cloudmusic;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adscounter.rtb.common.impl.RtbConfig;
import com.bxm.adscounter.rtb.common.utils.HttpClientUtils;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.warcar.utils.http.OkHttpUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;

import java.util.*;

/**
 * @author jingyu.li
 * @date 2023-11-21
 */
@Slf4j
public class CloudMusicRtbIntegration extends AbstractClickTrackerRtbIntegration {

    private final CloudMusicConfig config;

    public CloudMusicRtbIntegration(CloudMusicConfig config) {
        super(config,createHttpClient(config));
        this.config = config;
    }

    public static HttpClient createHttpClient(RtbConfig config) {
        int maxTotal = config.getMaxTotal();
        int defaultMaxPerRoute = config.getDefaultMaxPerRoute();
        int connectionRequestTimeout = config.getConnectionRequestTimeout();
        int connectTimeout = config.getConnectTimeout();
        int socketTimeout = config.getSocketTimeout();
        return HttpClientUtils.createHttpClient(maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout,"TLSv1.2");
    }

    @Override
    public Rtb rtb() {
        return Rtb.CloudMusic;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        KeyValueMap keyValueMap = request.getKeyValueMap();

        String clickId = keyValueMap.getFirst(Common.Param.CLICK_ID);

        if (StringUtils.isBlank(clickId)) {
            clickId = keyValueMap.getFirst(Common.Param.REFER);
            if (StringUtils.isBlank(clickId)) {
                throw new RtbIntegrationException(FailType.IllegalParameter, String.format("Cannot found callback from keyValueMap. keyValueMap: %s", JSONObject.toJSONString(keyValueMap)));
            }
        }

        Map<String, Object> params = new HashMap<>(15);
        params.put("platType", config.getPlatType());
        params.put("source", keyValueMap.getFirst("source_music"));
        params.put("event", request.getEventType());
        params.put("clickid", clickId);
        params.put("appkey", keyValueMap.getFirst("appkey"));
        params.put("timestamp", System.currentTimeMillis() / 1000);
        params.put("sign", getSign(params,keyValueMap.getFirst("appsecret")));

        String url = OkHttpUtils.appendParams(config.getUrl(), params);
        return new HttpGet(url);
    }

    private String getSign(Map<String, Object> params, String token) {
        Map<String, Object> sortParams = new TreeMap<>(params);
        Set<Map.Entry<String, Object>> set = sortParams.entrySet();
        StringBuilder valueSb = new StringBuilder();
        for (Map.Entry<String, Object> entry : set) {
            valueSb.append(token)
                    .append(entry.getKey())
                    .append(Objects.isNull(entry.getValue()) ? "" : entry.getValue());
        }
        valueSb.append(token);
        return DigestUtils.md5Hex(valueSb.toString()).toUpperCase();
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException {
        if (StringUtils.isBlank(body)) {
            return null;
        }

        JSONObject jsonObject = JSONObject.parseObject(body);
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        if (jsonObject.getInteger("code") == 200) {
            feedbackResponse.setSuccess(true);
        } else {
            feedbackResponse.setSuccess(false);
        }
        return feedbackResponse;
    }
}
