package com.bxm.adscounter.rtb.common.impl.huawei;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.base.Charsets;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.http.MediaType;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author tangxiao
 * @date 2022-12-02
 * @since 1.0
 */
@Slf4j
public class HuaweiOcpxRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public HuaweiOcpxRtbIntegration(HuaweiConfig config) {
        super(config, createHttpClient());
    }

    @Override
    public Rtb rtb() {
        return Rtb.HuaWeiOcpx;
    }

    @Override
    protected boolean isReadBodyForHttpResponse() {
        return true;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String url = getFeedbackUrl();
        String eventType = request.getEventType();

        if (StringUtils.isBlank(eventType)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'eventType'.");
        }

        KeyValueMap keyValueMap = request.getKeyValueMap();

        String oaid = keyValueMap.getFirst("oaid");
        String content_id = keyValueMap.getFirst("content_id");
        String campaign_id = keyValueMap.getFirst("campaign_id");
        String callback = keyValueMap.getFirst("callback");
        String token = keyValueMap.getFirst("token");

        if (StringUtils.isBlank(callback)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'callback'.");
        }
        if (StringUtils.isBlank(content_id)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'content_id'");
        }
        if (StringUtils.isBlank(campaign_id)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'campaign_id'");
        }

        // 两次解码后编码
        callback = UrlHelper.urlEncode(UrlHelper.urlDecode(UrlHelper.urlDecode(callback)));

        HuaweiRequest huaweiRequest = HuaweiRequest.builder()
                .callback(callback)
                .oaid(oaid)
                .conversion_type(eventType)
                .content_id(content_id)
                .campaign_id(campaign_id)
                .conversion_time(Objects.toString(System.currentTimeMillis() / 1000))
                .timestamp(Objects.toString(System.currentTimeMillis()))
                .build();

        String entity = JsonHelper.convert(huaweiRequest);

        HttpPost post = new HttpPost(url);
        post.setEntity(new StringEntity(entity, StandardCharsets.UTF_8));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        post.addHeader("Authorization", buildAuthorizationHeader(entity, token));
        return post;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException {
        Response response = JsonHelper.convert(body, Response.class);
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        if (Objects.nonNull(response)) {
            feedbackResponse.setSuccess(response.isSuccess());
            feedbackResponse.setBody(response.getResultMessage());
        } else {
            feedbackResponse.setSuccess(false);
        }
        return feedbackResponse;
    }


    private static CloseableHttpClient createHttpClient() {
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(2000).setConnectionRequestTimeout(1000).setConnectTimeout(1000).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
    }

    @Data
    @Builder
    @AllArgsConstructor
    public static class HuaweiRequest {
        private String oaid;
        private String conversion_type;
        private String content_id;
        private String callback;
        private String campaign_id;
        private String timestamp;
        private String conversion_time;
    }

    @Data
    public static class Response {
        private Integer resultCode;
        private String resultMessage;

        public boolean isSuccess() {
            return Objects.equals(resultCode, 0);
        }
    }

    /**
     * 计算请求头中的 Authorization
     *
     * @param body 消息体 json
     * @param key 秘钥
     * @return Authorization 鉴权头
     */
    public static String buildAuthorizationHeader(String body, String key) {
        // 请求头 header 中的 Authorization 格式
        final String authorizationFormat = "Digest validTime=\"{0}\", response=\"{1}\"";
        String authorization = null;
        try {
            byte[] keyBytes = key.getBytes(Charsets.UTF_8);
            byte[] bodyBytes = body.getBytes(Charsets.UTF_8);
            Mac mac = Mac.getInstance("HmacSHA256");
            SecretKey secretKey = new SecretKeySpec(keyBytes, "HmacSHA256");
            mac.init(secretKey);
            byte[] signatureBytes = mac.doFinal(bodyBytes);
            final String timestamp = String.valueOf(System.currentTimeMillis());
            final String signature = (signatureBytes == null) ? null : Hex.encodeHexString(signatureBytes);
            authorization = MessageFormat.format(authorizationFormat, timestamp, signature);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return authorization;
    }
}
