package com.bxm.adscounter.rtb.common.impl.kuaishou;

import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.rtb.common.DataFetchFailException;
import com.bxm.adscounter.rtb.common.DataFetcher;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.data.AdGroupData;
import com.bxm.adscounter.rtb.common.data.Parameter;
import com.bxm.adscounter.rtb.common.impl.AbstractHttpRtbIntegration;
import com.bxm.adsmanager.facade.model.base.Money;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.springframework.http.MediaType;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author allen
 * @date 2022-11-15
 * @since 1.0
 */
@Slf4j
public class KuaishouDataFetcher implements DataFetcher {

    private final KuaishouConfig config;
    private final JedisPool jedisPool;
    private final HttpClient httpClient;
    private final Fetcher fetcher;

    public KuaishouDataFetcher(KuaishouConfig config, JedisPool jedisPool, Fetcher fetcher) {
        this.config = config;
        this.httpClient = AbstractHttpRtbIntegration.createHttpClient(config);
        this.jedisPool = jedisPool;
        this.fetcher = fetcher;
    }

    private static String getToday() {
        return getToday(0);
    }

    private static String getToday(long days) {
        return LocalDate.now().plusDays(days).format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
    }

    private static String getHour(long hours) {
        return LocalDateTime.now().plusHours(hours).withMinute(0).format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"));
    }

    @Override
    public Rtb rtb() {
        return Rtb.Kuaishou;
    }

    @Override
    public List<AdGroupData> fetchCurrentHourData(Parameter parameter) throws DataFetchFailException {
        final String today = getToday();
        final String start = getHour(0);
        final String end = getHour(1);
        return fetch0(parameter, today, start, end);
    }

    @Override
    public List<AdGroupData> fetchPreviousHourData(Parameter parameter) throws DataFetchFailException {
        final String today = (LocalTime.now().getHour() == 0) ? getToday(-1) : getToday();
        final String start = getHour(-1);
        final String end = getHour(0);
        return fetch0(parameter, today, start, end);
    }

    private List<AdGroupData> fetch0(Parameter parameter, String date, String start, String end) throws DataFetchFailException {
        String tagId = parameter.getTagId();
        // advertiserId 先从缓存中获取
        String advertiserId = getAdvertiserId(tagId);
        String adGroupId = parameter.getAdGroupId();
        String shallowEventType = parameter.getShallowEventType();
        String deepEventType = parameter.getDeepEventType();
        Boolean needFetchKuaishouCpa = Optional.ofNullable(parameter.getNeedFetchKuaishouCpa()).orElse(false);

        String json = null;
        try {
            if (StringUtils.isBlank(advertiserId)) {
                throw new DataFetchFailException("'advertiserId' 不能为空，在广告位：" + tagId);
            }

            // 先使用GLOBAL获取，获取不到再使用广告位。
            KuaishouAk ak = getAk("GLOBAL", advertiserId);
            if (Objects.isNull(ak)) {
                ak = getAk(tagId, advertiserId);
            }

            if (Objects.isNull(ak)) {
                throw new DataFetchFailException(String.format("没有找到 AK 密钥，在广告位：%s 和 advertiserId：%s", tagId, advertiserId));
            }

            String accessToken = ak.getAccessToken();

            JSONObject params = new JSONObject();
            params.put("advertiser_id", advertiserId);
            params.put("start_date_min", start);
            params.put("end_date_min", end);
            params.put("start_date", date);
            params.put("end_date", date);
            params.put("temporal_granularity", "HOURLY");
            if (StringUtils.isNotBlank(adGroupId)) {
                params.put("unit_ids", new String[]{adGroupId});
            } else {
                params.put("page_size", 2000);
            }
            HttpPost post = new HttpPost(config.getUrlUnitReport());
            post.setEntity(new StringEntity(params.toJSONString()));
            post.addHeader("Access-Token", accessToken);
            post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);

            HttpResponse hr = httpClient.execute(post);
            json = EntityUtils.toString(hr.getEntity());
            KsUnitReportRes res = JSONObject.parseObject(json, KsUnitReportRes.class);
            if (!res.isSuccess()) {
                throw new DataFetchFailException(String.format("接口请求不成功！广告位：%s，advertiserId：%s。返回结果：%s", tagId, advertiserId, json));
            }

            List<KsUnitList> ksUnitLists = null;
            if (needFetchKuaishouCpa) {
                ksUnitLists = getUnitListPage(advertiserId, date, adGroupId, tagId, accessToken);
            }

            return convert(res, ksUnitLists, shallowEventType, deepEventType);
        } catch (IOException e) {
            throw new DataFetchFailException(String.format("接口网络异常（%s）！广告位：%s，advertiserId：%s。", e.getClass().getName(), tagId, advertiserId));
        } catch (JSONException e) {
            throw new DataFetchFailException(String.format("接口返回数据异常！广告位：%s，advertiserId：%s。返回结果：%s", tagId, advertiserId, json));
        }
    }

    private List<KsUnitList> getUnitListPage(String advertiserId, String date, String adGroupId, String tagId, String accessToken) throws DataFetchFailException, IOException {
        List<KsUnitList> totalData = new ArrayList<>(1000);
        int currentPage = 1;
        List<KsUnitList> currentPageData;
        // 分页查询
        do {
            if (log.isDebugEnabled()) {
                log.debug("[{}]执行中... 当前页 {}", getClass().getSimpleName(), currentPage);
            }
            currentPageData = getUnitListRes(advertiserId, date, adGroupId, tagId, accessToken, currentPage);
            currentPage++;
            if (CollectionUtils.isNotEmpty(currentPageData)) {
                totalData.addAll(currentPageData);
            }
        } while (CollectionUtils.isNotEmpty(currentPageData));
        return totalData;
    }

    private List<KsUnitList> getUnitListRes(String advertiserId, String date, String adGroupId, String tagId, String accessToken, Integer currentPage) throws IOException, DataFetchFailException {
        JSONObject params = new JSONObject();
        params.put("advertiser_id", advertiserId);
        params.put("page", currentPage);
        params.put("page_size", 500);
        if (StringUtils.isNotBlank(adGroupId)) {
            params.put("unit_ids", new String[]{adGroupId});
        }
        HttpPost post = new HttpPost(config.getUrlUnitList());
        post.setEntity(new StringEntity(params.toJSONString()));
        post.addHeader("Access-Token", accessToken);
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);

        HttpResponse hr = httpClient.execute(post);
        String json = EntityUtils.toString(hr.getEntity());
        KsUnitListRes res = JSONObject.parseObject(json, KsUnitListRes.class);
        if (!res.isSuccess()) {
            throw new DataFetchFailException(String.format("接口请求不成功！广告位：%s，advertiserId：%s。返回结果：%s", tagId, advertiserId, json));
        }
        if (Objects.nonNull(res.getData())) {
            return res.getData().getDetails();
        }
        return null;
    }

    private String getAppkey(String tagId) {
        if (StringUtils.isBlank(tagId)) {
            return null;
        }
        return StringUtils.split(tagId, "-")[0];
    }

    private String getAdvertiserId(String tagId) {
        return fetcher.hfetch(() -> KeyBuilder.build("rtb", "account_binding"), tagId, String.class);
    }

    private KuaishouAk getAk(String tagId, String advertiserId) {
        try (Jedis jedis = jedisPool.getResource()) {
            String v = jedis.hget(KeyBuilder.build("rtb", "conv", "AK", tagId), advertiserId);
            return Optional.ofNullable(v)
                    .filter(StringUtils::isNotBlank)
                    .map(s -> JsonHelper.convert(s, KuaishouAk.class))
                    .orElse(null);
        }
    }

    private List<AdGroupData> convert(KsUnitReportRes res, List<KsUnitList> ksUnitLists, String shallowEventType, String deepEventType) {
        if (!res.isSuccess()) {
            return null;
        }
        KsData<KsUnitReport> ksData = res.getData();
        if (Objects.isNull(ksData)) {
            return null;
        }
        List<KsUnitReport> details = ksData.getDetails();
        if (CollectionUtils.isEmpty(details)) {
            return null;
        }
        Map<String, Long> adGroupCpaMap = getAdGroupCpaMap(ksUnitLists);

        List<AdGroupData> ret = Lists.newArrayList();
        details.forEach(e -> {
            Long cpaBid = adGroupCpaMap.get(e.getUnit_id());
            AdGroupData adGroupData = new AdGroupData()
                    .setAdGroupId(e.getUnit_id())
                    .setDate(e.getStat_date())
                    .setHour(e.getStat_hour())
                    .setCharge(e.getCharge())
                    .setTime(System.currentTimeMillis())
                    .setCpa(Optional.ofNullable(cpaBid).map(cpa -> BigDecimal.valueOf(Money.ofLi(cpa).getYuan())).orElse(null));

            if (StringUtils.equals(shallowEventType, KuaishouConfig.EVENT_TYPE_FORM)) {
                adGroupData.setShallowConvCount(e.getForm_count());
                adGroupData.setShallowConvCost(e.getForm_cost());
            } else if (StringUtils.equals(shallowEventType, KuaishouConfig.EVENT_TYPE_USER_PAY)) {
                adGroupData.setShallowConvCount(e.getEvent_new_user_pay());
                adGroupData.setShallowConvCost(e.getEvent_new_user_pay_cost());
            } else if (StringUtils.equals(shallowEventType, KuaishouConfig.EVENT_TYPE_VALID_CLUES)) {
                adGroupData.setShallowConvCount(e.getEvent_valid_clues());
                adGroupData.setShallowConvCost(e.getEvent_valid_clues_cost());
            }

            if (StringUtils.equals(deepEventType, KuaishouConfig.EVENT_TYPE_FORM)) {
                adGroupData.setDeepConvCount(e.getForm_count());
                adGroupData.setDeepConvCost(e.getForm_cost());
            } else if (StringUtils.equals(deepEventType, KuaishouConfig.EVENT_TYPE_USER_PAY)) {
                adGroupData.setDeepConvCount(e.getEvent_new_user_pay());
                adGroupData.setDeepConvCost(e.getEvent_new_user_pay_cost());
            } else if (StringUtils.equals(deepEventType, KuaishouConfig.EVENT_TYPE_VALID_CLUES)) {
                adGroupData.setDeepConvCount(e.getEvent_valid_clues());
                adGroupData.setDeepConvCost(e.getEvent_valid_clues_cost());
            }

            adGroupData.setConvNumByImpression(e.getConversion_num_by_impression_7d());

            ret.add(adGroupData);
        });

        return ret;
    }

    private Map<String, Long> getAdGroupCpaMap(List<KsUnitList> ksUnitList) {
        if (CollectionUtils.isNotEmpty(ksUnitList)) {
                if (CollectionUtils.isNotEmpty(ksUnitList)) {
                    Map<String, Long> adGroupCpaMap = ksUnitList.stream()
                            .collect(Collectors.toMap(KsUnitList::getUnit_id, KsUnitList::getCpa_bid, (exist, replace) -> exist));
                    return adGroupCpaMap;
            }
        }
        return Collections.emptyMap();
    }

    public static class KsUnitReportRes extends Ks<KsUnitReport> {}
    public static class KsUnitListRes extends Ks<KsUnitList> {}

    @Data
    public static class Ks<T> {

        private int code = -1;
        private String message;
        private KsData<T> data;

        public boolean isSuccess() {
            return code == 0;
        }
    }

    @lombok.Data
    public static class KsData<T> {
        private int total_count;
        private List<T> details;
    }

    @lombok.Data
    public static class KsUnitReport {
        private String unit_id;
        private String stat_date;
        private Integer stat_hour;
        private Long form_count;
        private BigDecimal charge;
        private BigDecimal form_cost;
        private BigDecimal form_action_ratio;
        private Long event_new_user_pay;
        private BigDecimal event_new_user_pay_cost;
        private BigDecimal event_new_user_pay_ratio;
        private Long event_valid_clues;
        private BigDecimal event_valid_clues_cost;
        private Long conversion_num_by_impression_7d;
    }

    @lombok.Data
    public static class KsUnitList {
        private String unit_id;
        private Long cpa_bid;
    }

    public static void main(String[] args) {
        String today = getToday();
        System.out.println(today);
        String start = getHour(0);
        String end = getHour(1);
        System.out.println(start + " - " + end);

        System.out.println(getToday(-1));
        System.out.println(getHour(-1) + " - " + getHour(0));

        String today1 = (LocalTime.now().getHour() == 0) ? getToday(-1) : getToday();
        String start1 = getHour(-1);
        String end1 = getHour(0);

        System.out.println(today1 + " > " + start1 + " - " + end1);

        List<AdGroupData> list = new ArrayList<>();
        AdGroupData data = new AdGroupData()
                .setAdGroupId("8556006")
                .setDate("2022-11-16")
                .setHour(15)
                .setCharge(BigDecimal.valueOf(300))
                .setShallowConvCount(50L)
                .setShallowConvCost(BigDecimal.valueOf(30))
                .setDeepConvCount(20L)
                .setDeepConvCost(BigDecimal.valueOf(120))
                .setTime(System.currentTimeMillis());
        list.add(data);
        System.out.println(JsonHelper.convert(list));
    }

    public KeyGenerator hashKuaishouAdvertiserId(String tagId) {
        return () -> KeyBuilder.build("rtb", "kuaishou", "position_advertiserId");
    }
}
