package com.bxm.adscounter.rtb.common.impl.oppoOcpx;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.params.ProductionCommonParam;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.springframework.http.MediaType;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.util.HashMap;
import java.util.Map;

/**
 * @author jingyu.li
 * @date 2024-04-15
 */
@Slf4j
public class OppoOcpxRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String ADID = "ad_id";
    public static final String PKG = "package";

    private final OppoOcpxConfig config;

    public OppoOcpxRtbIntegration(OppoOcpxConfig config) {
        super(config);
        this.config = config;
    }

    @Override
    public Rtb rtb() {
        return Rtb.OPPOOCPX;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        String imei = keyValueMap.getFirst(ProductionCommonParam.IMEI_MD5);
        String oaid = keyValueMap.getFirst(ProductionCommonParam.OAID);
        if (checkNullImei(imei) && checkNullOaid(oaid)) {
            throw new RtbIntegrationException(FailType.IllegalParameter,"IMEI and OAID are both empty");
        }

        String adid = keyValueMap.getFirst(ADID);
        if (StringUtils.isBlank(adid) || "$ad$".equals(adid)) {
            throw new RtbIntegrationException(FailType.IllegalParameter,"adid is empty");
        }

        String pkg = keyValueMap.getFirst(PKG);
        if (StringUtils.isBlank(pkg)) {
            throw new RtbIntegrationException(FailType.IllegalParameter,"pkg is empty");
        }

        long timeMillis = System.currentTimeMillis();
        Map<String, Object> params = new HashMap<>();
        if (!checkNullImei(imei)) {
            params.put("imei",encode(imei));
        } else if (!checkNullOaid(oaid)) {
            params.put("ouid",encode(oaid));
        }
        params.put("timestamp",timeMillis);
        params.put(ADID,adid);
        params.put("dataType",request.getEventType());
        params.put("pkg",pkg);
        params.put("channel",1);
        params.put("ascribeType",1);
        String body = JsonHelper.convert(params);

        HttpPost post = new HttpPost(config.getUrl());
        post.addHeader("signature",sign(body,timeMillis));
        post.addHeader("timestamp",String.valueOf(timeMillis));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
        post.setEntity(new StringEntity(body,StandardCharsets.UTF_8));
        return post;
    }

    private boolean checkNullImei(String imei) {
        return StringUtils.isBlank(imei) || "$im$".equals(imei);
    }

    private boolean checkNullOaid(String oaid) {
        return StringUtils.isBlank(oaid) || "__OAID__".equals(oaid);
    }

    private String sign(String body,long timeMillis) {
        return DigestUtils.md5Hex(body + timeMillis + config.getSalt()).toLowerCase();
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) throws RtbIntegrationException {
        if (StringUtils.isBlank(body)) {
            return null;
        }

        JSONObject jsonObject = JSONObject.parseObject(body);
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        feedbackResponse.setSuccess(jsonObject.getInteger("ret") == 0);
        feedbackResponse.setBody(body);
        return feedbackResponse;
    }

    private String encode(String device) {
        byte[] encryptData = new byte[0];
        try {
            final Key dataKey = new SecretKeySpec(Base64.decodeBase64(config.getBase64Key()),
                    "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");
            cipher.init(Cipher.ENCRYPT_MODE, dataKey);
            encryptData = cipher.doFinal(device.getBytes(StandardCharsets.UTF_8));
        } catch (Exception e) {
            log.error("OPPOOCPX encode error:",e);
        }
        return Base64.encodeBase64String(encryptData).replaceAll("\r",
                "").replaceAll("\n", "");
    }
}
