package com.bxm.adscounter.rtb.common.impl.sigmobdsp;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.List;
import java.util.Objects;

/**
 * @author tangxiao
 * @date 2022-07-25
 * @since 1.0
 */
@Slf4j
public class SigmobDspRtbIntegration extends AbstractClickTrackerRtbIntegration {

    private static final String CALLBACK = "callback";

    public SigmobDspRtbIntegration(SigmobDspConfig config) {
        super(config);
    }

    @Override
    public Rtb rtb() {
        return Rtb.SigmobDsp;
    }

    @Override
    protected List<Integer> successHttpCode() {
        return Lists.newArrayList(HttpStatus.SC_OK, HttpStatus.SC_NO_CONTENT);
    }

    @Override
    protected boolean isReadBodyForHttpResponse() {
        return false;
    }

    @Override
    public String getClickIdParameterNameOnReferrer() {
        return "click_id";
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        String referrer = request.getReferrer();
        String eventType = request.getEventType();
        KeyValueMap keyValueMap = request.getKeyValueMap();
        String mt = keyValueMap.getMt();

        boolean isCallBackApi = Objects.equals(CALLBACK, eventType);

        MultiValueMap<String, String> queryParams = UriComponentsBuilder.fromUriString(referrer).build().getQueryParams();
        String callback = queryParams.getFirst("get_callback");
        String clickIdParameterNameOnReferrer = getClickIdParameterNameOnReferrer();
        String clickId = queryParams.getFirst(clickIdParameterNameOnReferrer);

        // 区分活动参与和其他类型
        if (isCallBackApi) {
            if (StringUtils.isBlank(callback)) {
                throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'get_callback' by referrer.");
            }
            String url = UrlHelper.urlDecode(callback);
            return new HttpGet(url);
        } else {
            if (StringUtils.isBlank(eventType)) {
                throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'eventType'.");
            }
            if (StringUtils.isBlank(clickId)) {
                throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found '" + clickIdParameterNameOnReferrer + "'.");
            }

            String url = UriComponentsBuilder.fromUriString(config.getUrl())
                    .replaceQueryParam("_uniq_key", "sig_active")
                    .replaceQueryParam("_ac_type", "80")
                    .replaceQueryParam("_user_id", "0")
                    .replaceQueryParam("platform", "0")
                    .replaceQueryParam("e", eventType)
                    .replaceQueryParam("clickid", clickId)
                    .replaceQueryParam("event_time", System.currentTimeMillis() / 1000)
                    .replaceQueryParam("currency_type", "CNY")
                    .replaceQueryParam("currency_amount", "0")
                    .toUriString();

            return new HttpGet(url);


        }
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        feedbackResponse.setSuccess(true);
        return feedbackResponse;
    }

}
