package com.bxm.adscounter.rtb.common.impl.sina;

import com.bxm.adscounter.model.constant.RedisKeyGenerator;
import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.consts.Inads;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.logging.log4j.util.Strings;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Map;
import java.util.Objects;

/**
 * @author tangxiao
 * @date 2024-04-15
 * @since 1.0
 */
@Slf4j
public class SinaRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "SET_D";
    private final Fetcher fetcher;


    public SinaRtbIntegration(SinaConfig config, Fetcher fetcher) {
        super(config);
        this.fetcher = fetcher;
    }

    @Override
    public Rtb rtb() {
        return Rtb.SINA;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        String clickId = UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
        if (StringUtils.isNotBlank(clickId)) {
            return clickId;
        }
        String key = clickEventLog.getIp() + clickEventLog.getUa();
        log.info("sina ipua : {}", key);
        if (StringUtils.isNotBlank(key)) {
            return fetcher.fetch(RedisKeyGenerator.strSinaIpUaClickId(key), String.class);
        }
        return Strings.EMPTY;
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        if (Objects.isNull(keyValueMap)) {
            return null;
        }
        return keyValueMap.getFirst(Inads.Param.TAGID);
    }

    @Override
    public boolean forceSaveClickTracker() {
        return true;
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        String clickId = getClickIdOnInadsAdClickLog(keyValueMap);
        if (StringUtils.isBlank(clickId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'click_id' by clickTracker.");
        }
        String token = request.getConfig().getToken();
        if (StringUtils.isBlank(token)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'token' by config.");
        }
        String cnvId = getCnvId(request);
        if (StringUtils.isBlank(cnvId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'cnvId' by config.");
        }

        Map<String, Object> params = Maps.newLinkedHashMap();
        params.put("cnv_id", cnvId);
        params.put("click_id", clickId);
        params.put("bhv_time", System.currentTimeMillis() / 1000);

        String feedbackUrl = getFeedbackUrl();
        feedbackUrl = OkHttpUtils.appendParams(feedbackUrl, params);
        feedbackUrl = feedbackUrl + "&sign=" + getSign(feedbackUrl, token);
        return new HttpGet(feedbackUrl);
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        return JsonHelper.convert(body, Response.class);
    }

    @Data
    private static class Response extends FeedbackResponse {
        private String code;
        private String info;

        @Override
        public boolean isSuccess() {
            return StringUtils.equals("600100", code);
        }
    }

    private String getCnvId(FeedbackRequest request) {
        PositionRtb positionRtb = request.getConfig();
        String extend = positionRtb.getExtend();
        if (StringUtils.isNotBlank(extend)) {
            CnvIdExtend cnvIdExtend = JsonHelper.convert(extend, CnvIdExtend.class);
            extend = request.isDeepConversion() ? cnvIdExtend.getDeepConvId() : cnvIdExtend.getShallowConvId();
        }
        return extend;
    }

    @Data
    private static class CnvIdExtend {
        private String shallowConvId;
        private String deepConvId;
    }

    private static String getSign(String feedbackUrl, String token) {
        try {
            Mac sha1Hmac = Mac.getInstance("HmacSHA1");
            SecretKeySpec secretKey = new SecretKeySpec(token.getBytes(StandardCharsets.UTF_8), "HmacSHA1");
            sha1Hmac.init(secretKey);
            byte[] hash = sha1Hmac.doFinal(feedbackUrl.getBytes(StandardCharsets.UTF_8));
            log.info("Signature: " + Base64.getUrlEncoder().encodeToString(hash));
            return Base64.getUrlEncoder().encodeToString(hash);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Strings.EMPTY;
    }
}
