package com.bxm.adscounter.rtb.common.impl.vivo;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Common;
import com.bxm.openlog.sdk.params.ProductionCommonParam;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UUIDHelper;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.logging.log4j.util.Strings;
import org.springframework.http.MediaType;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Optional;

/**
 * @author tangxiao
 * @date 2022-06-25
 * @since 1.0
 */
@Slf4j
public class VivoRtbIntegration extends AbstractClickTrackerRtbIntegration {

    public static final String CLICK_ID = "requestid";
    public static final String NAME_VIVO_SRC_TYPE = "vivo_src_type";

    public static final String SRC_TYPE_WEB = "Web";
    public static final String SRC_TYPE_APP = "APP";

    public VivoRtbIntegration(VivoConfig config) {
        super(config);
    }

    @Override
    public Rtb rtb() {
        return Rtb.Vivo;
    }

    @Override
    protected boolean isReadBodyForHttpResponse() {
        return true;
    }

    @Override
    public String getClickId(KeyValueMap clickTrackerKeyValueMap) {
        return clickTrackerKeyValueMap.getFirst(Common.Param.CLICK_ID);
    }

    @Override
    public String getClickIdOnInadsAdClickLog(KeyValueMap clickEventLog) {
        return UrlHelper.getFirstValueOfParamName(clickEventLog.getRef(), CLICK_ID);
    }

    @Override
    public String getAdGroupId(FeedbackRequest request) {
        String referrer = request.getReferrer();
        UriComponents build = UriComponentsBuilder.fromUriString(referrer).build();
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String clickId = queryParams.getFirst(CLICK_ID);
        KeyValueMap clickTracker = getClickTracker(clickId);

        return Optional.ofNullable(clickTracker)
                .map(k -> k.getFirst(Common.Param.AD_GROUP_ID))
                .orElse(StringUtils.EMPTY);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        KeyValueMap keyValueMap = request.getKeyValueMap();
        String referrer = request.getReferrer();
        String eventType = request.getEventType();

        PositionRtb config = request.getConfig();

        String vivoSrcId = config.getSourceId();
        String accessToken = config.getToken();
        String vivoAdvertiserId = Optional.ofNullable(config.getCustomerId()).orElse(Strings.EMPTY);

        // 从活动链接上加载参数
        MultiValueMap<String, String> queryParams =
                StringUtils.isBlank(referrer) ? new LinkedMultiValueMap<>() :
                UriComponentsBuilder.fromUriString(referrer).build().getQueryParams();

        // 同时兼容requestid click_id 和requestId
        String requestId = request.getClickId();
        requestId = StringUtils.isNotBlank(requestId) ? requestId : queryParams.getFirst("requestid");
        requestId = StringUtils.isNotBlank(requestId) ? requestId : queryParams.getFirst("requestId");

        // 从链接上获取adid 作为creativeId, 如果没有再试着获取链接上的creativeId
        String creativeId = queryParams.getFirst("adid");
        creativeId = StringUtils.isNotBlank(creativeId) ? creativeId : queryParams.getFirst("creativeId");
        creativeId = StringUtils.isNotBlank(creativeId) ? creativeId : keyValueMap.getFirst("creativeid");

        if (StringUtils.isBlank(vivoSrcId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'vivoSrcId'.");
        }
        if (StringUtils.isBlank(accessToken)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'accessToken'.");
        }

        if (StringUtils.isBlank(requestId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'requestId' or 'requestid' by referrer.");
        }
        if (StringUtils.isBlank(creativeId)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'creativeId' or 'adid' by referrer.");
        }
        if (StringUtils.isBlank(eventType)) {
            throw new RtbIntegrationException(FailType.IllegalParameter, "Cannot found 'eventType'.");
        }

        // 如果 vivo_src_type 为空，则默认 Web
        String srcType = StringUtils.defaultIfBlank(keyValueMap.getFirst(NAME_VIVO_SRC_TYPE), SRC_TYPE_WEB);

        VivoRequest.Data data = new VivoRequest.Data();
        data.setCvType(eventType);
        data.setCvTime(System.currentTimeMillis());
        data.setRequestId(requestId);
        data.setCreativeId(creativeId);

        //如果转化类型是单次付费并且支付金额不为空，回传参数extParam
        if (StringUtils.equals("PAY_ONETIME",request.getEventType()) && StringUtils.isNotBlank(keyValueMap.getFirst("pay_amount"))) {
            HashMap<String, String> extParam = Maps.newHashMap();
            extParam.put("payAmount", keyValueMap.getFirst("pay_amount"));
            data.setExtParam(extParam);
        }

        VivoRequest.VivoRequestBuilder builder = VivoRequest.builder();

        if (StringUtils.equals(srcType, SRC_TYPE_APP)) {
            builder.pkgName(keyValueMap.getFirst("pkgname"));

            String oaid = keyValueMap.getFirst(ProductionCommonParam.OAID);
            String oaidMd5 = keyValueMap.getFirst(ProductionCommonParam.OAID_MD5);
            String imei = keyValueMap.getFirst(ProductionCommonParam.IMEI);
            String imeiMd5 = keyValueMap.getFirst(ProductionCommonParam.IMEI_MD5);
            if (StringUtils.isNotBlank(oaid)) {
                data.setUserIdType("OAID");
                data.setUserId(oaid);
            } else if (StringUtils.isNotBlank(oaidMd5)) {
                data.setUserIdType("OAID_MD5");
                data.setUserId(oaidMd5);
            } else if (StringUtils.isNotBlank(imei)) {
                data.setUserIdType("IMEI");
                data.setUserId(imei);
            } else if (StringUtils.isNotBlank(imeiMd5)) {
                data.setUserIdType("IMEI_MD5");
                data.setUserId(imeiMd5);
            }
        } else {
            builder.pageUrl(referrer);
        }

        VivoRequest vivoRequest = builder
                .srcId(vivoSrcId)
                .srcType(srcType)
                .dataList(Lists.newArrayList(data))
                .build();

        String entity = JsonHelper.convert(vivoRequest);
        if (log.isInfoEnabled()) {
            log.info("[vivo] json body : {}", entity);
        }

        String url = getFeedbackUrl();
        url = UriComponentsBuilder.fromUriString(url)
                .replaceQueryParam("access_token", accessToken)
                .replaceQueryParam("timestamp", System.currentTimeMillis())
                .replaceQueryParam("nonce", UUIDHelper.generate())
                .replaceQueryParam("advertiser_id", vivoAdvertiserId).toUriString();

        HttpPost post = new HttpPost(url);
        post.setEntity(new StringEntity(entity, StandardCharsets.UTF_8));
        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        request.setRequestBody(entity);
        return post;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        VivoResponse response = JsonHelper.convert(body, VivoResponse.class);
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        feedbackResponse.setSuccess(response.isSuccess());
        feedbackResponse.setBody(response.getMessage());
        return feedbackResponse;
    }

    @Override
    public String[] getMissingAdGroupIdList() {
        return new String[] {"__CREATIVEID__", "NULL"};
    }

    @Data
    private static class VivoResponse {
        private String code;
        private String message;

        public boolean isSuccess() {
            return StringUtils.equalsIgnoreCase("0", code);
        }
    }
}
