package com.bxm.adscounter.rtb.common.impl.weibodownload;

import com.bxm.adscounter.rtb.common.FailType;
import com.bxm.adscounter.rtb.common.Rtb;
import com.bxm.adscounter.rtb.common.RtbIntegrationException;
import com.bxm.adscounter.rtb.common.feedback.FeedbackRequest;
import com.bxm.adscounter.rtb.common.feedback.FeedbackResponse;
import com.bxm.adscounter.rtb.common.impl.AbstractClickTrackerRtbIntegration;
import com.bxm.adsprod.facade.ticket.rtb.PositionRtb;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.http.OkHttpUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;

import java.util.Map;
import java.util.Objects;

/**
 * 微博应用下载回传
 * @author lijingyu
 * @date 2023-02-15
 * @since 1.0
 */
@Slf4j
public class WeiBoDownloadRtbIntegration extends AbstractClickTrackerRtbIntegration {
    private static final int EXPIRE_TIME_SECOND = 24 * 60 * 60;
    private Updater updater;
    private Fetcher fetcher;

    public WeiBoDownloadRtbIntegration(WeiBoDownloadConfig config) {
        super(config, createHttpClient());
    }

    private static CloseableHttpClient createHttpClient() {
        RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(4000)
                .setConnectionRequestTimeout(1000)
                .setConnectTimeout(1000).build();
        return HttpClientBuilder.create().setDefaultRequestConfig(requestConfig).build();
    }

    /**
     * 监测到点击的时候，把点击监测链接上的参数保存下来，因为回传的时候要用（因为用户唯一标识不确定是哪一个，所以都存了一遍）
     * @param clickTrackerKeyValueMap
     */
    @Override
    public void saveClickTracker(KeyValueMap clickTrackerKeyValueMap) {
        String idfaMd5 = clickTrackerKeyValueMap.getFirst("idfa_md5");
        String imeiMd5 = clickTrackerKeyValueMap.getFirst("imei_md5");
        String oaidMd5 = clickTrackerKeyValueMap.getFirst("oaid_md5");
        String androididMd5 = clickTrackerKeyValueMap.getFirst("androidid_md5");
        String ip = clickTrackerKeyValueMap.getFirst("ip");
        String ua = clickTrackerKeyValueMap.getFirst("ua");
        if (StringUtils.isNotBlank(idfaMd5)) {
            updater.update(createKey(idfaMd5), clickTrackerKeyValueMap, EXPIRE_TIME_SECOND);
        }
        if (StringUtils.isNotBlank(imeiMd5)) {
            updater.update(createKey(imeiMd5), clickTrackerKeyValueMap, EXPIRE_TIME_SECOND);
        }
        if (StringUtils.isNotBlank(oaidMd5)) {
            updater.update(createKey(oaidMd5), clickTrackerKeyValueMap, EXPIRE_TIME_SECOND);
        }
        if (StringUtils.isNotBlank(androididMd5)) {
            updater.update(createKey(androididMd5), clickTrackerKeyValueMap, EXPIRE_TIME_SECOND);
        }
        if (StringUtils.isNotBlank(ip) && StringUtils.isNotBlank(ua)) {
            updater.update(createKey(ip + ua), clickTrackerKeyValueMap, EXPIRE_TIME_SECOND);
        }
    }

    @Override
    public KeyValueMap getClickTracker(String clickId) {
        return fetcher.fetch(createKey(clickId),KeyValueMap.class);
    }

    @Override
    protected HttpRequestBase create(FeedbackRequest request) throws RtbIntegrationException {
        PositionRtb config = request.getConfig();
        String company = config.getCustomerId();
        if (StringUtils.isBlank(company)) {
            throw new RtbIntegrationException(FailType.IllegalParameter,"company:Configuration cannot be empty");
        }
        KeyValueMap keyValueMap = request.getKeyValueMap();
        String idfaMd5 = keyValueMap.getFirst("idfa_md5");
        String imeiMd5 = keyValueMap.getFirst("imei_md5");
        String oaidMd5 = keyValueMap.getFirst("oaid_md5");
        String androididMd5 = keyValueMap.getFirst("androidid_md5");
        String ip = keyValueMap.getFirst("ip");
        String ua = keyValueMap.getFirst("ua");
        KeyValueMap clickTracker = null;
        boolean isAvailableDevice = true;
        if (StringUtils.isNotBlank(idfaMd5)) {
            clickTracker = getClickTracker(idfaMd5);
            isAvailableDevice = false;
        }
        if (StringUtils.isNotBlank(imeiMd5)) {
            clickTracker = getClickTracker(imeiMd5);
            isAvailableDevice = false;
        }
        if (StringUtils.isNotBlank(oaidMd5)) {
            clickTracker = getClickTracker(oaidMd5);
            isAvailableDevice = false;
        }
        if (StringUtils.isNotBlank(androididMd5)) {
            clickTracker = getClickTracker(androididMd5);
            isAvailableDevice = false;
        }
        if (StringUtils.isNotBlank(ip) && StringUtils.isNotBlank(ua)) {
            clickTracker = getClickTracker(ip + ua);
            isAvailableDevice = false;
        }
        if (isAvailableDevice) {
            throw new RtbIntegrationException(FailType.NotFoundClickId,"No device number available!");
        }

        if (Objects.isNull(clickTracker)) {
            throw new RtbIntegrationException(FailType.NotFoundClickId,"Click monitoring parameters cannot be found!");
        }

        String imp = clickTracker.getFirst("IMP");
        if (StringUtils.isBlank(imp)) {
            throw new RtbIntegrationException(FailType.NotFoundClickId,"Click monitoring parameters IMP cannot be found!");
        }

        String url = getFeedbackUrl();
        Map<String, Object> params = Maps.newHashMap();
        params.put("company", company);
        params.put("action_type", request.getEventType());
        params.put("IMP", imp);
        if (StringUtils.isNotBlank(imeiMd5)) {
            params.put("imei", imeiMd5);
        }
        if (StringUtils.isNotBlank(oaidMd5)) {
            params.put("oaid", oaidMd5);
        }
        if (StringUtils.isNotBlank(idfaMd5)) {
            params.put("idfa", idfaMd5);
        }
        if (StringUtils.isNotBlank(ip) && StringUtils.isNotBlank(ua)) {
            params.put("ip", ip);
            params.put("ua", ua);
        }
        if (StringUtils.isNotBlank(clickTracker.getFirst("MODEL"))) {
            params.put("model",clickTracker.getFirst("MODEL"));
        }
        if (StringUtils.isNotBlank(clickTracker.getFirst("LANGUAGE"))) {
            params.put("language",clickTracker.getFirst("LANGUAGE"));
        }
        if (StringUtils.isNotBlank(clickTracker.getFirst("GX_CAID"))) {
            params.put("GX_CAID",clickTracker.getFirst("GX_CAID"));
        }

        url = OkHttpUtils.appendParams(url, params);
        log.info("微博应用下载回传url：{}",url);
        HttpGet httpGet = new HttpGet(url);
        return httpGet;
    }

    @Override
    public Rtb rtb() {
        return Rtb.WEIBODOWNLOAD;
    }

    @Override
    protected FeedbackResponse convert(FeedbackRequest request, String body) {
        FeedbackResponse feedbackResponse = new FeedbackResponse();
        feedbackResponse.setSuccess(true);
        return feedbackResponse;
    }
}
