package com.bxm.adscounter.temp.scheduler;

import com.bxm.openlog.extension.client.Fallback;
import com.bxm.openlog.extension.client.HttpClientUtils;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.utils.NamedThreadFactory;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.springframework.util.ClassUtils;

import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2022-10-18
 * @since 1.0
 */
@Slf4j
public class MyOpenLogClient implements OpenLogClient {

    private final ThreadPoolExecutor executor;
    private final HttpClient httpClient;

    private Timer timer;

    private MyOpenLogClient(ThreadPoolExecutor executor, int maxTotal, int defaultMaxPerRoute, int connectionRequestTimeout, int connectTimeout, int socketTimeout) {
        this.executor = executor;
        this.httpClient = HttpClientUtils.createHttpClient(maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
    }

    public static MyOpenLogClient.Builder builder() {
        return new MyOpenLogClient.Builder();
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        Class<?> clazz = ClassUtils.getUserClass(this);
        registerGauge(registry, this, clazz);
        registerTimer(registry, this, clazz);
    }

    @Override
    public boolean request(String url) throws IOException {
        return this.request2(url, null);
    }

    @Override
    public void asyncRequest(String url) {
        this.asyncRequest(url, null);
    }

    @Override
    public void asyncRequest(String url, Consumer<Fallback> fallback) {
        this.asyncRequest2(url, null, fallback);
    }

    public void asyncRequest2(String url, Map<String, String> headers, Consumer<Fallback> fallback) {
        this.executor.execute(() -> {
            try {
                if (!request2(url, headers)) {
                    if (Objects.nonNull(fallback)) {
                        fallback.accept(Fallback.builder().url(url).build());
                    }
                }
            } catch (IOException e) {
                if (Objects.nonNull(fallback)) {
                    fallback.accept(Fallback.builder().url(url).exception(e).build());
                }
            }
        });
    }

    private boolean request2(String url, Map<String, String> headers) throws IOException {
        long start = System.nanoTime();
        HttpGet get = null;
        try {
            get = new HttpGet(url);
            if (MapUtils.isNotEmpty(headers)) {
                for (Map.Entry<String, String> entry : headers.entrySet()) {
                    get.addHeader(entry.getKey(), entry.getValue());
                }
            }
            HttpResponse response = httpClient.execute(get);
            return response.getStatusLine().getStatusCode() == HttpStatus.SC_OK;
        } finally {
            if (Objects.nonNull(get)) {
                get.releaseConnection();
            }
            if (Objects.nonNull(timer)) {
                timer.record((System.nanoTime() - start), TimeUnit.NANOSECONDS);
            }
        }
    }

    @Override
    public int queueSize() {
        return executor.getQueue().size();
    }

    @Override
    public void close() {
        this.executor.shutdown();
        try {
            if (!this.executor.awaitTermination(30, TimeUnit.SECONDS)) {
                log.warn("This executor was forced terminated!");
            }
        } catch (InterruptedException e) {
            log.error("awaitTermination: ", e);
        }
    }

    public HttpClient getHttpClient() {
        return httpClient;
    }

    public ThreadPoolExecutor getExecutor() {
        return executor;
    }

    private void registerGauge(MeterRegistry registry, OpenLogClient client, Class<?> clazz) {
        Gauge.builder("OpenLogClient.queue", 0, value -> client.queueSize())
                .tags("name", clazz.getName())
                .register(registry);
    }

    private void registerTimer(MeterRegistry registry, OpenLogClient client, Class<?> clazz) {
        this.timer = Timer.builder("OpenLogClient.timer")
                .tag("name", clazz.getName())
                .register(registry);
        log.info("Registering OpenLogClient {} timer meter successful.", client);
    }

    public static class Builder {

        private int maxTotal = 200;
        private int defaultMaxPerRoute = 20;
        private int connectionRequestTimeout = 100;
        private int connectTimeout = 200;
        private int socketTimeout = 500;
        private ThreadPoolExecutor executor;

        {
            int processors = Runtime.getRuntime().availableProcessors();
            int max = 10000;
            executor = new ThreadPoolExecutor(processors, processors, 0, TimeUnit.SECONDS,
                    new LinkedBlockingQueue<>(max), new NamedThreadFactory("openlog-client"));
        }

        private Builder() {}

        public MyOpenLogClient build() {
            return new MyOpenLogClient(executor, maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
        }

        public MyOpenLogClient.Builder setMaxTotal(int maxTotal) {
            this.maxTotal = maxTotal;
            return this;
        }

        public MyOpenLogClient.Builder setDefaultMaxPerRoute(int defaultMaxPerRoute) {
            this.defaultMaxPerRoute = defaultMaxPerRoute;
            return this;
        }

        public MyOpenLogClient.Builder setConnectionRequestTimeout(int connectionRequestTimeout) {
            this.connectionRequestTimeout = connectionRequestTimeout;
            return this;
        }

        public MyOpenLogClient.Builder setConnectTimeout(int connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public MyOpenLogClient.Builder setSocketTimeout(int socketTimeout) {
            this.socketTimeout = socketTimeout;
            return this;
        }

        public MyOpenLogClient.Builder setExecutor(ThreadPoolExecutor executor) {
            this.executor = executor;
            return this;
        }
    }
}
