/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the
 * confidential and proprietary information of bianxianmao.com ("Confidential
 * Information"). You shall not disclose such Confidential Information and shall
 * use it only in accordance with the terms of the license agreement you entered
 * into with bianxianmao.com.
 */
package com.bxm.adsmanager.facade.model.base;

import com.google.common.collect.Lists;
import com.google.common.primitives.Longs;

import java.io.Serializable;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Iterator;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * <h3>金额类</h3>
 * <p>
 * 	用于封装涉及到金额表示和计算的逻辑，以厘为基础计量单位，1厘(li）= 0.1分(cent) <br/>
 *  支持: 厘(long)、分(long)、元(double)  
 * </p>
 * @version $Id: Money.java, v1.0.0 2017年11月14日 下午4:38:39, huangkai
 */
public class Money implements Serializable, Comparable<Money> {
    private static final long serialVersionUID = 7013286615982402339L;
    /** 金额, 精确到厘  */
	private long li;

	/** 舍入模式 */
	private RoundingMode roundingMode = RoundingMode.HALF_UP;

	public final static Money ZERO = Money.ofLi(0L);

	private Money() {}
	
	/** 通过厘构造Money类 */
	public static Money ofLi(long li) {
		Money money = new Money();
		money.setLi(li);
		return money;
	}
	
	/** 通过分构造Money类 */
	public static Money ofCent(long cent) {
		Money money = new Money();
		money.setLi(convertCentToLi(cent));
		return money;
	}
	
	/** 通过元构造Money类, 默认舍入模式为：RoundingMode.HALF_UP */
	public static Money ofYuan(double yuan) {
		return ofYuan(yuan, RoundingMode.HALF_UP);
	}

	/** 通过元构造Money类 */
	public static Money ofYuan(double yuan, RoundingMode roundingMode) {
		Money money = new Money();
		money.setLi(convertYuanToLi(yuan, roundingMode));
		return money;
	}

	/** 设置舍入模式，默认为: RoundingMode.HALF_UP */
	public Money withRoundingMode(RoundingMode roundingMode) {
		this.roundingMode = roundingMode;
		return this;
	}

	/** 取得额度值，以厘为单位 */
	public long getLi() {
		return this.li;
	}
	
	/** 金额转化为分 */
	public long getCent() {
		return BigDecimal.valueOf(getLi())
					.movePointLeft(1)
					.setScale(0, this.roundingMode)
					.longValue();
	}
	
	/** 金额转化为元, 精确到厘 */
	public double getYuan() {
		return BigDecimal.valueOf(getLi())
					.movePointLeft(3)
					.setScale(3, this.roundingMode) //精确到厘
					.doubleValue();
	}

	/**
	 * 计算总金额
	 * 
	 * @param monies 需要计算总金额的Money实例
	 * @return 包含总金额的Money结果实例
	 */
	public static Money total(Money... monies) {
		if (monies.length == 0) {
            throw new IllegalArgumentException("Money array must not be empty");
        }
        return total(Lists.newArrayList(monies));
	}
	
	/**
	 * 计算总金额
	 * 
	 * @param monies 需要计算总金额的Money实例
	 * @return 包含总金额的Money结果实例
	 */
	public static Money total(Iterable<? extends Money> monies) {
        checkNotNull(monies, "Money iterator must not be null");

        Iterator<? extends Money> it = monies.iterator();
        if (it.hasNext() == false) {
            throw new IllegalArgumentException("Money iterator must not be empty");
        }
        Money total = it.next();
        checkNotNull(total, "Money iterator must not contain null entries");
        while (it.hasNext()) {
            total = total.plus(it.next());
        }
        return total;
    }

	/** 
	 * 增加金额
	 * 
	 * @param money 包含要增加的额度的Money实例
	 * @return 增加完额度后的Money实例
	 */
	public Money plus(Money money) {
		checkNotNull(money, "money must not be null");
		return plus(money.getLi());
	}
	/**
	 * 以厘的形式增加金额
	 * 
	 * @param li 要增加的额度，以厘表示
	 * @return 增加完额度后的Money实例
	 */
	public Money plus(long li) {
		setLi(getLi() + li);
		return this;
	}
	
	/**
	 * 以分的形式增加金额
	 * 
	 * @param cent 要增加的额度，以分表示
	 * @return 增加完额度后的Money实例
	 */
	public Money plusCent(long cent) {
		if(cent == 0L) { return this; }

		plus(convertCentToLi(cent));
		return this;
	}
	
	/**
	 * 以元的形式增加金额
	 * 
	 * @param yuan 要增加的额度，以元表示
	 * @return  增加完额度后的Money实例
	 */
	public Money plusYuan(double yuan) {
		if(yuan == 0.0) { return this; }

		plus(convertYuanToLi(yuan, this.roundingMode));
		return this;
	}

	/**
	 * 减少金额
	 * 
	 * @param money 包含要减少的额度的Money实例
	 * @return 减少金额后的Money实例
	 */
	public Money minus(Money money) {
		checkNotNull(money, "money must not be null");
		return minus(money.getLi());
	}

	/**
	 * 以厘的形式减少金额
	 * 
	 * @param li 要减少的额度，以厘表示
	 * @return 减少金额后的Money实例
	 */
	public Money minus(long li) {
		if(li == 0L) { return this; }

		setLi(getLi() - li);
		return this;
	}
	
	/**
	 * 以分的形式减少金额
	 * 
	 * @param cent 要减少的额度，以分钱表示
	 * @return 减少金额后的Money实例
	 */
	public Money minusCent(long cent) {
		if(cent == 0L) { return this; }

		setLi(getLi() - convertCentToLi(cent));
		return this;
	}

	/**
	 * 以元的形式减少金额
	 * 
	 * @param yuan 要减少的额度，以元钱表示
	 * @return 减少金额后的Money实例
	 */
	public Money minusYuan(double yuan) {
		if (yuan == 0.0) { return this; }

		setLi(getLi() - convertYuanToLi(yuan, this.roundingMode));
		return this;
	}
	
	/**
	 * 以指定倍数放大金额
	 * 
	 * @param valueToMultiplyBy 金额放大的倍数
	 * @return 进行乘法计算后的Moeny结果实例
	 */
	public Money multipliedBy(long valueToMultiplyBy) {
		if (valueToMultiplyBy == 1) { return this; }

		setLi(getLi() * valueToMultiplyBy); 
		return this;
	}
	
	/**
	 * 以指定倍数放大金额
	 * 
	 * @param valueToMultiplyBy 金额放大的倍数
	 * @return 进行乘法计算后的Moeny结果实例
	 */
	public Money multipliedBy(double valueToMultiplyBy) {
		return multipliedBy(valueToMultiplyBy, this.roundingMode);
	}

	/**
	 * 以指定倍数放大金额
	 * 
	 * @param valueToMultiplyBy 金额放大的倍数
	 * @param roundingMode 舍位模式
	 * @return 进行乘法计算后的Moeny结果实例
	 */
	public Money multipliedBy(double valueToMultiplyBy, RoundingMode roundingMode) {
		if (valueToMultiplyBy == 1.0) { return this; }

		BigDecimal amountLiBig = BigDecimal.valueOf(getLi());
		BigDecimal valueToMultiplyByBig = BigDecimal.valueOf(valueToMultiplyBy);
		BigDecimal result = amountLiBig.multiply(valueToMultiplyByBig).setScale(0, roundingMode);
		setLi(result.longValue());

		return this;
	}
	
	/** 判断该Money实例的金额是否为负 */
	public boolean isNegative() {
		return getLi() < 0;
	}

	/** 判断该Money实例的金额是否为负或者为零 */
	public boolean isNegativeOrZero() {
		return getLi() <= 0;
	}
	
	/** 判断两个Money实例是否额度相等 */
	public boolean isEqual(Money other) {
        return compareTo(other) == 0;
    }

	/** 判断Money实例的金额是否大于要比较的Money实例 */
    public boolean isGreaterThan(Money other) {
        return compareTo(other) > 0;
    }

    /** 判断Money实例的金额是否小于要比较的Money实例 */
    public boolean isLessThan(Money other) {
        return compareTo(other) < 0;
    }

	/** 将金额从元钱转为厘 */
	private static long convertYuanToLi(double yuan, RoundingMode roundingMode) {
		return BigDecimal.valueOf(yuan)
					.movePointRight(3)
					.setScale(0, roundingMode)
					.longValue();
	}

	/** 将金额从分钱转为厘 */
	private static long convertCentToLi(long cent) {
		return cent * 10;
	}
	
	/** li，单位为厘 */
	private void setLi(long li) {
		this.li = li;
	}

    @Override
	public int hashCode() {
		return Longs.hashCode(getLi());
	}
	 
	@Override
    public boolean equals(Object other) {
        if (this == other) { return true; }

        if (other instanceof Money) {
            Money otherMoney = (Money) other;
            return getLi() == otherMoney.getLi();
        }

        return false;
    }

	@Override
	public int compareTo(Money otherMoney) {
		if(otherMoney == null) { return 1; }

		return Longs.compare(getLi(), otherMoney.getLi());
	}
	
	@Override
	public String toString() {
		return getLi() + " li";
	}
}
