package com.bxm.adsmedia.web.controller.media;

import java.util.List;

import javax.validation.constraints.NotNull;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.baomidou.mybatisplus.plugins.Page;
import com.bxm.adsmedia.common.context.user.UserSessionContext;
import com.bxm.adsmedia.common.exception.BusinessException;
import com.bxm.adsmedia.facade.enumdata.MediaTypeEnum;
import com.bxm.adsmedia.model.dto.media.UpdateMediaDTO;
import com.bxm.adsmedia.model.vo.media.MediaInfoVO;
import com.bxm.adsmedia.model.vo.media.MediaNameAndIdVO;
import com.bxm.adsmedia.model.vo.media.MediaVO;
import com.bxm.adsmedia.service.media.MediaService;
import com.bxm.warcar.utils.response.ResultModel;
import com.bxm.warcar.utils.response.ResultModelFactory;

/**
 * Description 媒体
 *
 * @author hxpeng
 * <p>
 * Date 2019/1/17 17:10
 */
@Validated
@RestController
@RequestMapping("/media")
public class MediaController {


    @Autowired
    private MediaService mediaService;

    /**
     * Description: 查询媒体列表
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/17 17:57
     *
     * @param idOrName 媒体名称或ID
     * @return com.bxm.warcar.utils.response.ResultModel<com.bxm.adsmedia.model.vo.media.MediaVO>
     * @author hxpeng
     */
    @RequestMapping(value = "/page", method = RequestMethod.GET)
    public ResultModel<Page<MediaVO>> getPage(@RequestParam(name = "idOrName", required = false) String idOrName,
                                              @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNum,
                                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        return ResultModelFactory.SUCCESS(mediaService.getPage(idOrName, pageNum, pageSize));
    }


    /**
     * Description: 查询媒体列表
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/17 17:57
     *
     * @return com.bxm.warcar.utils.response.ResultModel<com.bxm.adsmedia.model.vo.media.MediaVO>
     * @author hxpeng
     */
    @RequestMapping(value = "/getMediaNameById", method = RequestMethod.GET)
    public ResultModel<List<MediaNameAndIdVO>> getMediaNameById() {
        return ResultModelFactory.SUCCESS(mediaService.getMediaNameById(UserSessionContext.getProviderId()));
    }

    /**
     * Description: 获取媒体信息，根据主键
     * JDK version used:<JDK1.8>
     * Create Date：2019/2/13 11:29
     *
     * @param id 主键
     * @return com.bxm.warcar.utils.response.ResultModel<com.bxm.adsmedia.model.vo.media.MediaInfoVO>
     * @author hxpeng
     */
    @RequestMapping(value = "/get", method = RequestMethod.GET)
    public ResultModel<MediaInfoVO> get(@NotNull(message = "id不能为空！") @RequestParam(name = "id") Long id) {
        return ResultModelFactory.SUCCESS(mediaService.get(id));
    }

    /**
     * Description: 获取当前登录用户所有有效得媒体列表
     * JDK version used:<JDK1.8>
     * Create Date：2019/2/13 16:27
     *
     * @return com.bxm.warcar.utils.response.ResultModel<java.util.List<com.bxm.adsmedia.model.vo.media.MediaVO>>
     * @author hxpeng
     */
    @RequestMapping(value = "/getAllEffective", method = RequestMethod.GET)
    public ResultModel<List<MediaVO>> getAllEffective() {
        return ResultModelFactory.SUCCESS(mediaService.getAllEffective(UserSessionContext.getProviderId()));
    }


    /**
     * Description: 媒体 新增
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/17 17:37
     *
     * @param dto 参数
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/add", method = RequestMethod.POST)
    public ResultModel<Boolean> add(@Validated @RequestBody UpdateMediaDTO dto) {
        if (null == MediaTypeEnum.getMediaTypeNameByCode(dto.getMediaType())) {
            throw new BusinessException("请输入正确的媒体系统类型！");
        }
        this.validateWebSite(dto);
        return ResultModelFactory.SUCCESS(mediaService.add(UserSessionContext.getProviderId(), dto));
    }

    private void validateWebSite(UpdateMediaDTO dto) {
        if (MediaTypeEnum.H5.getCode().equals(dto.getMediaType())) {
            if ( StringUtils.isBlank(dto.getWebsiteUrl())) {
                throw new BusinessException("网址域名不能为空！");
            }
            dto.setWebsiteUrl(dto.getWebsiteUrl().trim());
            int maxLength = 500;
            if (dto.getWebsiteUrl().length() > maxLength) {
                throw new BusinessException("网址域名不能超过五百个字符！");
            }
        }
    }

    /**
     * Description: 媒体 修改(初审失败，终审失败调用edit)
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/17 17:37
     *
     * @param dto 参数
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/edit", method = RequestMethod.POST)
    public ResultModel<Boolean> edit(@Validated @RequestBody UpdateMediaDTO dto) {
        if (null == dto.getMediaId()) {
            throw new BusinessException("id不能为空！");
        }
        if (null == MediaTypeEnum.getMediaTypeNameByCode(dto.getMediaType())) {
            throw new BusinessException("请输入正确的媒体系统类型！");
        }
        this.validateWebSite(dto);
        return ResultModelFactory.SUCCESS(mediaService.update(dto));
    }
    /**
     * //todo  产品设计的不严谨，审核不通的广告位和审核通过的广告位的必传参数不一样
     * Description: 媒体 修改(审核通过的广告位调用这个接口)(通过，初审状态调用edit1 ;)
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/17 17:37
     *
     * @param dto 参数
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/edit1", method = RequestMethod.POST)
    public ResultModel<Boolean> edit1(@Validated @RequestBody UpdateMediaDTO dto) {
        if (null == dto.getMediaId()) {
            throw new BusinessException("id不能为空！");
        }
        if (null == MediaTypeEnum.getMediaTypeNameByCode(dto.getMediaType())) {
            throw new BusinessException("请输入正确的媒体系统类型！");
        }
        dto.setFirstStatus(true);
        return ResultModelFactory.SUCCESS(mediaService.update(dto));
    }
    /**
     * Description: 判断开发者是否有过审的媒体
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/18 18:47
     *
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/checkHasApprovedMedia", method = RequestMethod.GET)
    public ResultModel<Boolean> checkHasApprovedMedia() {
        return ResultModelFactory.SUCCESS(mediaService.checkHasApprovedMedia(UserSessionContext.getProviderId()));
    }


}
