package com.bxm.adsmedia.web.controller.provider;

import javax.validation.constraints.NotBlank;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.bxm.adsmedia.common.context.user.UserSessionContext;
import com.bxm.adsmedia.common.exception.BusinessException;
import com.bxm.adsmedia.common.util.RegExpValidatorUtils;
import com.bxm.adsmedia.model.dto.auth.UserRegisterDTO;
import com.bxm.adsmedia.model.dto.provider.UpdateProviderDTO;
import com.bxm.adsmedia.model.enums.CaptchaSceneEnum;
import com.bxm.adsmedia.model.enums.ProviderTypeEnum;
import com.bxm.adsmedia.model.vo.auth.UserVO;
import com.bxm.adsmedia.model.vo.provider.ProviderBaseInfoVO;
import com.bxm.adsmedia.service.common.CaptchaService;
import com.bxm.adsmedia.service.common.CommonService;
import com.bxm.adsmedia.service.provider.ProviderService;
import com.bxm.warcar.utils.response.ResultModel;
import com.bxm.warcar.utils.response.ResultModelFactory;

import lombok.extern.slf4j.Slf4j;


/**
 * Description 开发者
 *
 * @author hxpeng
 * <p>
 * Date 2019/1/15 10:08
 */
@Validated
@Slf4j
@RestController
@RequestMapping("/provider")
public class ProviderController {

    @Autowired
    private ProviderService providerService;
    @Autowired
    private CaptchaService captchaService;
    @Autowired
    private CommonService commonService;

    /**
     * Description: 登录
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 14:46
     *
     * @param uuid    前端传的客户端标识
     * @param uName   用户名(邮箱/手机号)
     * @param pwd     密码
     * @param captcha 验证码
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/login", method = RequestMethod.POST)
    public ResultModel<UserVO> login(@NotBlank(message = "uuid不能为空！")
                                     @RequestHeader(name = "uuid") String uuid,

                                     @NotBlank(message = "用户名不能为空！")
                                     @RequestParam(name = "uName") String uName,

                                     @NotBlank(message = "密码不能为空！")
                                     @RequestParam(name = "uPwd") String pwd,

                                     @NotBlank(message = "验证码不能为空！")
                                     @RequestParam(name = "captcha") String captcha) {
        if (!captchaService.verifyImgCaptcha(CaptchaSceneEnum.LOGIN, uuid, captcha)) {
            throw new BusinessException("验证码错误！");
        }
        return ResultModelFactory.SUCCESS(providerService.login(uuid, uName, pwd, captcha));
    }

    /**
     * Description: 退出登录
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/22 15:04
     *
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/loginOut", method = RequestMethod.POST)
    public ResultModel<Boolean> loginOut() {
        commonService.removeUserToken();
        return ResultModelFactory.SUCCESS(true);
    }

    /**
     * Description: 注册-短信验证码
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/22 14:04
     *
     * @param uuid     前端传的客户端标识
     * @param captcha  验证码
     * @param phoneNum 手机号
     * @return void
     * @author hxpeng
     */
    @RequestMapping(value = "/register/smsCaptcha", method = RequestMethod.GET)
    public ResultModel<Boolean> registerSmsCaptchaGet(@NotBlank(message = "uuid不能为空！")
                                                      @RequestHeader(name = "uuid") String uuid,

                                                      @NotBlank(message = "验证码不能为空！")
                                                      @RequestParam(name = "captcha") String captcha,

                                                      @NotBlank(message = "手机号不能为空！")
                                                      @RequestParam(name = "phoneNum") String phoneNum) {
        if (!captchaService.verifyImgCaptcha(CaptchaSceneEnum.REGISTER, uuid, captcha)) {
            throw new BusinessException("请输入正确的图片验证码！");
        }
        if (providerService.checkPhoneNumExist(phoneNum)) {
            throw new BusinessException("手机号已被注册！");
        }
        return ResultModelFactory.SUCCESS(captchaService.smsCaptcha(CaptchaSceneEnum.REGISTER, phoneNum, "【变现猫】变现猫开发者平台欢迎您的加入,获取的验证码是："));
    }

    /**
     * Description: 注册
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 16:33
     *
     * @param dto 注册参数
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/register", method = RequestMethod.POST)
    public ResultModel<Boolean> register(@Validated @RequestBody UserRegisterDTO dto) {
        if (!captchaService.verifySmsCaptcha(CaptchaSceneEnum.REGISTER, dto.getPhoneNum(), dto.getCaptcha())) {
            return ResultModelFactory.FAILED400("注册失败，验证码输入错误！");
        }
        // 校验邮箱
        if (!RegExpValidatorUtils.isEmail(dto.getEmail())) {
            return ResultModelFactory.FAILED400("注册失败，邮箱格式不正确！");
        }
        // 校验手机号
        if (!RegExpValidatorUtils.isMobileNO(dto.getPhoneNum())) {
            return ResultModelFactory.FAILED400("注册失败，手机号格式不正确！");
        }
        // 校验开发者类型
        if (null == ProviderTypeEnum.getByCode(dto.getProviderTypeCode())) {
            return ResultModelFactory.FAILED400("注册失败，请选择正确的开发者类型！");
        }
        providerService.register(dto);
        return ResultModelFactory.SUCCESS(true);
    }

    /**
     * Description: 校验邮箱是否已存在
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 16:45
     *
     * @param email 邮箱
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/checkEmailExist", method = RequestMethod.GET)
    public ResultModel<Boolean> checkEmailExist(@NotBlank(message = "邮箱不能为空！") @RequestParam(name = "email") String email) {
        return ResultModelFactory.SUCCESS(providerService.checkEmailExist(email));
    }

    /**
     * Description: 校验手机号是否已存在
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/16 14:53
     *
     * @param phoneNum 手机号
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/checkPhoneNumExist", method = RequestMethod.GET)
    public ResultModel<Boolean> checkPhoneNumExist(@NotBlank(message = "手机号不能为空！") @RequestParam(name = "phoneNum") String phoneNum) {
        return ResultModelFactory.SUCCESS(providerService.checkPhoneNumExist(phoneNum));
    }

    /**
     * Description: 修改密码 当登录的时候 （需要旧密码 和 新密码）
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/16 19:15
     *
     * @param oldPwd 旧密码
     * @param newPwd 新密码
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
//    @RequestMapping(value = "/updatePwd", method = RequestMethod.POST)
//    public ResultModel<Boolean> updatePwdWhenLoggedIn(@RequestParam(name = "oldPwd") String oldPwd,
//                                                      @RequestParam(name = "newPwd") String newPwd) {
//        return ResultModelFactory.SUCCESS(providerService.updatePwdWhenLoggedIn(oldPwd, newPwd, UserSessionContext.getProviderId()));
//    }

    /**
     * Description: 忘记密码-短信验证码
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/22 14:04
     *
     * @param uuid     前端传的客户端标识
     * @param captcha  图片验证码
     * @param phoneNum 手机号
     * @return void
     * @author hxpeng
     */
    @RequestMapping(value = "/forgetPwd/smsCaptcha", method = RequestMethod.GET)
    public ResultModel<Boolean> forgetPwdSmsCaptchaGet(@NotBlank(message = "uuid不能为空！")
                                                       @RequestHeader(name = "uuid") String uuid,

                                                       @NotBlank(message = "验证码不能为空！")
                                                       @RequestParam(name = "captcha") String captcha,

                                                       @NotBlank(message = "手机号不能为空！")
                                                       @RequestParam(name = "phoneNum") String phoneNum) {
        if (!captchaService.verifyImgCaptcha(CaptchaSceneEnum.UPDATE_PWD, uuid, captcha)) {
            throw new BusinessException("请输入正确的图片验证码！");
        }
        return ResultModelFactory.SUCCESS(captchaService.smsCaptcha(CaptchaSceneEnum.UPDATE_PWD, phoneNum, "【变现猫】您当前正在修改密码，获取的验证码是："));
    }


    /**
     * Description: 修改密码 当忘记密码的时候 （需要注册手机号 和 验证码 和 新密码）
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/16 19:16
     *
     * @param newPwd   新密码
     * @param phoneNum 注册手机号
     * @param captcha  短信验证码
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/forgetPwd", method = RequestMethod.POST)
    public ResultModel<Boolean> forgetPwd(@NotBlank(message = "新密码不能为空！")
                                          @RequestParam(name = "newPwd") String newPwd,

                                          @NotBlank(message = "手机号不能为空！")
                                          @RequestParam(name = "phoneNum") String phoneNum,

                                          @NotBlank(message = "验证码不能为空！")
                                          @RequestParam(name = "captcha") String captcha) {
        // 校验验证码
        if (!captchaService.verifySmsCaptcha(CaptchaSceneEnum.UPDATE_PWD, phoneNum, captcha)) {
            throw new BusinessException("修改失败，验证码输入错误！");
        }
        return ResultModelFactory.SUCCESS(providerService.updatePwdWhenForget(newPwd, phoneNum));
    }

    /**
     * Description: 修改手机号短信验证码
     * JDK version used:<JDK1.8>
     * Create Date：2019/2/16 14:46

     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/updatePhoneNum/smsCaptcha", method = RequestMethod.GET)
    public ResultModel<Boolean> updatePhoneNumSmsCaptchaGet() {
        return ResultModelFactory.SUCCESS(captchaService.smsCaptcha(CaptchaSceneEnum.UPDATE_PHONE_NUM, UserSessionContext.getPhoneNum(), "【变现猫】您当前正在修改手机号，获取的验证码是："));
    }

    /**
     * Description: 修改手机号
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/16 19:17
     *
     * @param newPhoneNum 手机号码
     * @param captcha     验证码
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
//    @RequestMapping(value = "/updatePhoneNum", method = RequestMethod.POST)
//    public ResultModel<Boolean> updatePhoneNum(@RequestParam(name = "newPhoneNum") String newPhoneNum,
//                                               @RequestParam(name = "captcha") String captcha) {
//        // 校验验证码
//        if (!captchaService.verifySmsCaptcha(CaptchaSceneEnum.UPDATE_PHONE_NUM, UserSessionContext.getPhoneNum(), captcha)) {
//            throw new BusinessException("修改失败，验证码输入错误！");
//        }
//        if (providerService.checkPhoneNumExist(newPhoneNum)) {
//            throw new BusinessException("修改失败，修改的手机号已被注册！");
//        }
//        return ResultModelFactory.SUCCESS(providerService.updatePhoneNum(UserSessionContext.getProviderId(), newPhoneNum));
//    }

    /**
     * Description: 获取基本信息
     * JDK version used:<JDK1.8>
     * Create Date：2019/2/12 15:16
     *
     * @return com.bxm.warcar.utils.response.ResultModel<com.bxm.adsmedia.model.vo.provider.ProviderBaseInfoVO>
     * @author hxpeng
     */
    @RequestMapping(value = "/getBaseInfo", method = RequestMethod.GET)
    public ResultModel<ProviderBaseInfoVO> getBaseInfo() {
        return ResultModelFactory.SUCCESS(providerService.getBaseInfo(UserSessionContext.getProviderId()));
    }

    /**
     * Description: 修改基本信息
     * * JDK version used:<JDK1.8>
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/16 19:19
     *
     * @param dto  修改参数
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author hxpeng
     */
    @RequestMapping(value = "/updateBaseInfo", method = RequestMethod.POST)
    public ResultModel<Boolean> updateBaseInfo(@RequestBody UpdateProviderDTO dto) {
        dto.setId(UserSessionContext.getProviderId());
        return ResultModelFactory.SUCCESS(providerService.updateBaseInfo(dto));
    }

    /**
     * Description: 校验开发者财务信息是否存在
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/22 16:37
     *
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.String>
     * @author hxpeng
     */
    @RequestMapping(value = "/getFinanceInfoStatus", method = RequestMethod.GET)
    public ResultModel<String> getFinanceInfoStatus() {
        return ResultModelFactory.SUCCESS(providerService.getFinanceInfoStatus(UserSessionContext.getProviderId()));
    }


    /**
     * Description: 提交审核信息
     * * JDK version used:<JDK1.8>
     * JDK version used:<JDK1.8>
     * Create Date：2019/7/02 16:50
     *
     * @return com.bxm.warcar.utils.response.ResultModel<java.lang.Boolean>
     * @author zhengwangeng
     */
    @RequestMapping(value = "/updateReviewInfo", method = RequestMethod.POST)
    public ResultModel<Boolean> updateReviewInfo() {
        return ResultModelFactory.SUCCESS(providerService.updateReviewInfo(UserSessionContext.getProviderId()));
    }

}
