package com.bxm.adsmedia.web.handler;

import com.bxm.adsmedia.common.exception.AuthException;
import com.bxm.adsmedia.common.exception.BusinessException;
import com.bxm.warcar.utils.response.ResultModel;
import com.bxm.warcar.utils.response.ResultModelFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MaxUploadSizeExceededException;

import javax.validation.ConstraintViolationException;


/**
 * Description 全局异常捕捉
 *
 * @author hxpeng
 * <p>
 * Date 2018/9/10 15:14
 */
@Slf4j
@ControllerAdvice
public class GlobalExceptionHandler {

    /**
     * Description: 缺少必要参数 抛得异常
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 14:50
     *
     * @param e 异常
     * @return com.bxm.warcar.utils.response.ResultModel
     * @author hxpeng
     */
    @ExceptionHandler(ConstraintViolationException.class)
    @ResponseBody
    public ResultModel handleRequestParameterException(ConstraintViolationException e) {
        log.error("异常信息" + e.getMessage(),e);
        return ResultModelFactory.FAILED("400", e.getMessage());
    }

    @ExceptionHandler(MissingServletRequestParameterException.class)
    @ResponseBody
    public ResultModel handleRequestParameterException(MissingServletRequestParameterException e) {
        log.error("异常信息" + e.getMessage(),e);
        return ResultModelFactory.FAILED("400", e.getParameterName() + "：请检查参数是否正确！");
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    @ResponseBody
    public ResultModel handleMethodArgumentNotValidException(MethodArgumentNotValidException e) {
        log.error("异常信息" + e.getMessage(),e);
        return ResultModelFactory.FAILED("400", e.getBindingResult().getFieldErrors().get(0).getDefaultMessage());
    }


    /**
     * Description: 缺少必要参数 抛得异常
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 14:50
     *
     * @param e 异常
     * @return com.bxm.warcar.utils.response.ResultModel
     * @author hxpeng
     */
    @ExceptionHandler(ServletRequestBindingException.class)
    @ResponseBody
    public ResultModel handleRequestBindingException(ServletRequestBindingException e) {
        log.error("请求异常：", e);
        return ResultModelFactory.FAILED("400", "请检查参数是否正确！");
    }

    /**
     * Description: 图片上传过大
     * JDK version used:<JDK1.8>
     * Create Date：2019/3/14 11:49
     *
     * @param e
     * @return com.bxm.warcar.utils.response.ResultModel
     * @author hxpeng
     */
    @ExceptionHandler(MaxUploadSizeExceededException.class)
    @ResponseBody
    public ResultModel handleMaxUploadSizeExceededException(MaxUploadSizeExceededException e) {
        log.error("异常信息" + e.getMessage(),e);
        return ResultModelFactory.FAILED("400", "上传失败，上传文件过大！");
    }

    /**
     * Description: 未catch住的异常
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 14:55
     *
     * @param e 异常
     * @return com.bxm.warcar.utils.response.ResultModel
     * @author hxpeng
     */
    @ExceptionHandler(Exception.class)
    @ResponseBody
    public ResultModel handleException(Exception e) {
        log.error("【system error】", e);
        return ResultModelFactory.FAILED("500", e.getMessage());
    }

    /**
     * 运行时异常
     */
    @ExceptionHandler(RuntimeException.class)
    @ResponseBody
    public ResultModel handleRuntimeException(RuntimeException e) {
        log.error("异常信息" + e.getMessage(),e);
        return ResultModelFactory.FAILED("500", e.getMessage());
    }


    /**
     * Description: 业务异常
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 14:55
     *
     * @param e 异常
     * @return com.bxm.warcar.utils.response.ResultModel
     * @author hxpeng
     */
    @ExceptionHandler(BusinessException.class)
    @ResponseBody
    public ResultModel handleBusinessException(BusinessException e) {
        log.error("异常信息" + e.getMessage(),e);
        return ResultModelFactory.FAILED("500", e.getMessage());
    }


    /**
     * Description: 无权限异常
     * JDK version used:<JDK1.8>
     * Create Date：2019/1/15 14:58
     *
     * @param e 异常
     * @return com.bxm.warcar.utils.response.ResultModel
     * @author hxpeng
     */
    @ExceptionHandler(AuthException.class)
    @ResponseBody
    public ResultModel handleAuthException(AuthException e) {
        log.error("异常信息" + e.getMessage(),e);
        return ResultModelFactory.FAILED("401", StringUtils.isBlank(e.getMessage()) ? "未拥有权限操作！" : e.getMessage());
    }


}
