package com.bxm.adsprod.facade.ticket;

import com.bxm.adsprod.facade.commons.CachePushableFields;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import org.apache.commons.lang.math.NumberUtils;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Map;

/**
 * <h3>广告券相关KEY创建者</h3>
 *
 * @author allen
 * @since V1.0.0 2017/12/12
 */
public final class TicketKeyGenerator {

    public static class Filter {
        private Filter() {}

        public static KeyGenerator getRegionDefined() {
            return () -> KeyBuilder.build("AD", "FILTER", "REGION", "DEFINED");
        }

        public static KeyGenerator getRegionUndefined() {
            return () -> KeyBuilder.build("AD", "FILTER", "REGION", "UNDEFINED");
        }

        public static KeyGenerator getPositionGroup(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_GROUP", ticketId);
        }

        public static KeyGenerator getPositionDefinedBlackList() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "DEFINED", "BLACKLIST");
        }

        public static KeyGenerator getPositionDefinedWhiteList() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "DEFINED", "WHITELIST");
        }

        public static KeyGenerator getPositionUndefined() {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION", "UNDEFINED");
        }

        public static KeyGenerator getTimeline(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "FILTER", "TIMELINE", ticketId);
        }
    }

    public static class Advertiser {

        private Advertiser() {}

        public static KeyGenerator getBalance() {
            return () -> KeyBuilder.build("AD", "ADVERTISER", "BALANCE");
        }
    }

    public static class Media {
        private Media() {}

        public static KeyGenerator getPositions() {
            return () -> KeyBuilder.build("AD", "POSITION", "ALL");
        }

        public static KeyGenerator getPositionDomain(Map<String, Object> parameters) {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_DOMAIN",
                    parameters.get(CachePushableFields.POSITION_ID),
                    parameters.get(CachePushableFields.DOMAIN_CODE));
        }

        public static KeyGenerator getPositionTag(Map<String, Object> parameters) {
            return () -> KeyBuilder.build("AD", "FILTER", "POSITION_TAG",
                    parameters.get(CachePushableFields.POSITION_ID),
                    parameters.get(CachePushableFields.TAG_CODE));
        }
    }

    public static class User {
        private User() {}

        public static KeyGenerator getUser(String uid) {
            return () -> KeyBuilder.build("AD", "RECORD", "LAST", uid);
        }
    }

    public static class Statistics {

        private Statistics() {}

        public static KeyGenerator getBudgetOfDaily() {
            return getBudgetOfDaily(DateHelper.format("yyyyMMdd"));
        }

        public static KeyGenerator getBudgetOfDaily(String date) {
            return () -> KeyBuilder.build("AD", "COUNTER", "DAILY_BUDGET", date);
        }

        public static KeyGenerator getClickOfDaily() {
            return getClickOfDaily(DateHelper.format("yyyyMMdd"));
        }

        public static KeyGenerator getClickOfDaily(String date) {
            return () -> KeyBuilder.build("AD", "COUNTER", "CLICK",
                    date);
        }

        public static KeyGenerator getViewOfDaily() {
            return getViewOfDaily(DateHelper.format("yyyyMMdd"));
        }

        public static KeyGenerator getViewOfDaily(String date) {
            return () -> KeyBuilder.build("AD", "COUNTER", "VIEW",
                    date);
        }

        public static KeyGenerator getPositionViewOfDaily(String position) {
            return getPositionViewOfDaily(DateHelper.format("yyyyMMdd"), position);
        }

        public static KeyGenerator getPositionViewOfDaily(String date, String position) {
            return () -> KeyBuilder.build("AD", "COUNTER", "VIEW",
                    date, position);
        }

        public static KeyGenerator getBudgetOfHourly() {
            return getBudgetOfHourly(NumberUtils.toInt(DateHelper.format("H"), -1));
        }

        public static KeyGenerator getBudgetOfHourly(int hour) {
            return () -> KeyBuilder.build("AD", "COUNTER", "HOURLY_BUDGET",
                    DateHelper.format("yyyyMMdd"),
                    hour);
        }

        public static KeyGenerator getViewOfHourly() {
            return getViewOfHourly(NumberUtils.toInt(DateHelper.format("H"), -1));
        }

        public static KeyGenerator getViewOfHourly(int hour) {
            return () -> KeyBuilder.build("AD", "COUNTER", "HOURLY_VIEW",
                    DateHelper.format("yyyyMMdd"),
                    hour);
        }

        public static KeyGenerator getClickOfDaily(BigInteger ticketId) {
            return getClickOfDaily(DateHelper.getDate(), ticketId);
        }

        public static KeyGenerator getClickOfDaily(String date, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UDAILY_CLICK",
                    date,
                    ticketId);
        }

        public static KeyGenerator getViewOfDaily(BigInteger ticketId) {
            return getViewOfDaily(DateHelper.getDate(), ticketId);
        }

        public static KeyGenerator getViewOfDaily(String date, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UDAILY_VIEW",
                    date,
                    ticketId);
        }

        public static KeyGenerator getClickOfWeekly(BigInteger ticketId) {
            return getClickOfWeekly(DateHelper.format("yyyyw"), ticketId);
        }

        public static KeyGenerator getClickOfWeekly(String dateweek, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UWEEKLY_CLICK",
                    dateweek,
                    ticketId);
        }

        public static KeyGenerator getViewOfWeekly(BigInteger ticketId) {
            return getViewOfWeekly(DateHelper.format("yyyyw"), ticketId);
        }

        public static KeyGenerator getViewOfWeekly(String dateweek, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "UWEEKLY_VIEW",
                    dateweek,
                    ticketId);
        }

        public static KeyGenerator getBudgetOfPositionGroup(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "COUNTER", "DAILY_BUDGET_P_G", DateHelper.format("yyyyMMdd"), ticketId);
        }
    }

    public static class Weight {

        @Deprecated
        public static KeyGenerator getWeight(BigInteger ticketId, String positionId) {
            return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET", ticketId);
        }

        public static KeyGenerator getWeight(String positionId) {
            return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET", positionId);
        }
    }

    public static class Scene {

        private Scene() {}

        public static KeyGenerator getScene(int scene) {
            return () -> KeyBuilder.build("AD", "TICKET", "SCENE", scene);
        }

        public static KeyGenerator getSceneViews(int scene, String uid) {
            return () -> KeyBuilder.build("AD", "COUNTER", "SCENE", scene, uid);
        }
    }

    public static class Ocpc {

        public static KeyGenerator getConsume(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "CONSUME", ticketId);
        }

        public static KeyGenerator getClick(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "CLICK", ticketId);
        }

        @Deprecated
        public static KeyGenerator getValidClick(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "VALIDCLICK", ticketId);
        }

        public static KeyGenerator getValidClick(BigInteger ticketId, String position) {
            return () -> KeyBuilder.build("AD", "OCPC", "VALIDCLICK", ticketId, position);
        }

        public static KeyGenerator getClosed(String position) {
            return () -> KeyBuilder.build("AD", "OCPC", "CLOSED", position);
        }

        public static KeyGenerator getFloor() {
            return () -> KeyBuilder.build("AD", "OCPC", "FLOOR");
        }

        public static KeyGenerator getOfferSheetLock(String position, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "OFFER_SHEET_LOCK", position, ticketId);
        }
        public static KeyGenerator getOfferSheet(String position, BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "OCPC", "OFFER_SHEET", position, ticketId);
        }
    }

    public static class Temp {

        private Temp() {}

        public static KeyGenerator getTimelineOutOfBudget(BigInteger ticketId) {
            return () -> KeyBuilder.build("AD", "TEMP", "TIMELINE", "OUT_OF_BUDGET", ticketId);
        }

        public static KeyGenerator getTickets(int os, int app, String regionCode, String position) {
            return () -> KeyBuilder.build("AD", "TEMP", "TICKET", os, app, regionCode, position);
        }
    }

    public static KeyGenerator getUser(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "USER",
                parameters.get(CachePushableFields.IMEI));
    }

    public static KeyGenerator getTimes(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "FILTER", "TIMES",
                parameters.get(CachePushableFields.TICKET_ID));
    }

    public static KeyGenerator getTicketAssets() {
        return () -> KeyBuilder.build("AD", "TICKET", "ASSETS");
    }

    public static KeyGenerator getAvailableTickets() {
        return () -> KeyBuilder.build("AD", "TICKET", "AVAILABLE");
    }

    public static KeyGenerator getAllTickets() {
        return () -> KeyBuilder.build("AD", "TICKET", "ALL");
    }

    public static KeyGenerator getTicketPrice(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PRICE",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }

    public static KeyGenerator getTicketPriceOfTodayFirst(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PRICE_TF",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }
    public static KeyGenerator getTicketPriceFloatPercent(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PRICE_FP",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }

    public static KeyGenerator getTicketMif(Map<String, Object> parameters) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_MIF",
                parameters.get(CachePushableFields.TICKET_ID),
                parameters.get(CachePushableFields.POSITION_ID));
    }

    public static KeyGenerator getTicketProfitForCpa(LocalDate date) {
        return () -> KeyBuilder.build("AD", "WEIGHT", "TICKET_PROFIT_FOR_CPA", date.format(DateTimeFormatter.ofPattern("yyyyMMdd")));
    }
}
