package com.bxm.adsprod.third.controller;

import com.bxm.adsprod.third.constant.Constants;
import com.bxm.adsprod.third.constant.ErrorCode;
import com.bxm.adsprod.third.model.HeiNiuSubmitRsp;
import com.bxm.adsprod.third.model.LandUserInfo;
import com.bxm.adsprod.third.model.LandVo;
import com.bxm.adsprod.third.model.Result;
import com.bxm.adsprod.third.model.land.BaseLandUserParamsDto;
import com.bxm.adsprod.third.model.land.LandUserActionResultVo;
import com.bxm.adsprod.third.service.LandService;

import com.bxm.adsprod.third.util.IpUtils;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**  
* @Title: LandController
* @Description:  落地页服务
* @author youhaijun
* @date 2018年6月7日
*/  
@RestController
@RequestMapping("/land")
public class LandController extends BaseController {

	@Autowired
	private LandService landService;

	/**
	 * 根据用户获取黑牛保险落地页素材
	 * @param channel @see LandChannel
	 * @return
	 */
	@RequestMapping(value = "/getHeiNiuLandInfo", method = RequestMethod.GET)
	public Result<LandVo> getHeiNiuLandInfo(String channel, HttpServletRequest request, HttpServletResponse response){
		try {
			response.setHeader("Access-Control-Allow-Origin", "*");
			String ip = IpUtils.getRemoteIp(request);
			if(StringUtils.isBlank(ip)){
				//校验ip存在
				return failResult(ErrorCode.E0100000);
			}
			return successResult(landService.getHeiNiuLandInfo(channel, ip));
		}catch (Exception e){
			return failResult(e);
		}
	}

    /**
     * 请求短信验证码服务
     * @param userPhone
     * @param channel 落地页标识，通过落地页id做base64脱敏
     * @return
     */
    @RequestMapping(value = "/smsVerifyCode", method = RequestMethod.GET)
    public Result<Boolean> smsVerifyCode(String channel, String userPhone, HttpServletRequest request, HttpServletResponse response){
		try {
			response.setHeader("Access-Control-Allow-Origin", "*");
			Preconditions.checkNotNull(channel, "参数[channel]不存在");
			Preconditions.checkNotNull(userPhone, "参数[userPhone]不存在");
			String ip = IpUtils.getRemoteIp(request);
			if(StringUtils.isBlank(ip)){
				//校验ip存在
				return failResult(ErrorCode.E0100000);
			}
			return successResult(landService.smsVerifyCode(channel, userPhone, ip));
		}catch (Exception e){
			return failResult(e);
		}
    }

	/**
	 * 提交落地页表单
	 * @return
	 */
	@RequestMapping(value = "/submit", method = {RequestMethod.GET, RequestMethod.POST})
	public Result<HeiNiuSubmitRsp> submit(LandUserInfo landUserInfo, HttpServletRequest request, HttpServletResponse response){
		try {
			response.setHeader("Access-Control-Allow-Origin", "*");
			Preconditions.checkNotNull(landUserInfo.getOrderId(), "参数[orderId]不存在");
			Preconditions.checkNotNull(landUserInfo.getVerifyCode(), "参数[verifyCode]不存在");
			String ip = IpUtils.getRemoteIp(request);
			String ua = request.getHeader("User-Agent");
			if(StringUtils.isBlank(ip) || StringUtils.isBlank(ua)){
				//校验ip存在
				return failResult(ErrorCode.E0100000);
			}
			return successResult(landService.submit(landUserInfo, ip, ua));
		}catch (Exception e){
			return failResult(e);
		}
	}

	@RequestMapping(value = "/setCache", method = {RequestMethod.GET, RequestMethod.POST})
	public Result<Boolean> setCache(String id){
		try {
			return successResult(landService.setIpDefined(id));
		}catch (Exception e){
			return failResult(e);
		}
	}




    /**
     * Description: 提交落地页表单(腾讯新闻)
     * JDK version used: <JDK1.8>
     * Author： hxpeng
     * Create Date： 2018/7/18 20:12
     */
    @RequestMapping(value = "/tencentnews/submit", method = {RequestMethod.GET, RequestMethod.POST})
    public Result<LandUserActionResultVo> submitInfo(String paramsJson, HttpServletRequest request, HttpServletResponse response){
        try {
            response.setHeader("Access-Control-Allow-Origin", "*");
            String ip = IpUtils.getRemoteIp(request);
            String ua = request.getHeader("User-Agent");
            if(StringUtils.isBlank(ip) || StringUtils.isBlank(ua)){
                //校验ip存在
                return failResult(ErrorCode.E0100000);
            }
            return successResult(landService.submitInfo(new BaseLandUserParamsDto(ip, ua, Constants.LandUserInfoActionCode.TENCENT_NEWS, paramsJson)));
        }catch (Exception e){
            return failResult(e);
        }
    }
}
