package com.bxm.adx.facade.constant.redis;

import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;

/**
 * ADX相关的redis key<br/>
 *
 * @author kerry.jiang
 * @date 2020/3/25 19:41
 */
public final class AdxKeyGenerator {

    /** 全局配置信息 **/
    private final static String GLOBAL_CONFIG = "ADX:GLOBAL:CONFIG";

    /**
     * 全局配置信息
     */
    public static KeyGenerator getConfig() {
        return () -> GLOBAL_CONFIG;
    }

    /**
     * 设备信息
     */
    public static class Device {
        
        /**
         * SDK初始化上报信息
         */
        public static KeyGenerator getInitInfo(String devNo) {
            return () -> KeyBuilder.build("ADX", "SDK", "INIT", "devNo", devNo);
        }
        
        /**
         * 已安装的APP列表(包名)
         */
        public static KeyGenerator getInitInstalledList(String devNo) {
            return () -> KeyBuilder.build("ADX", "SDK", "INIT", "devNo", devNo, "APPS");
        }
    }

    /**
     * 开发者应用信息
     */
    public static class App {

        /**
         * 应用信息
         */
        public static KeyGenerator getInfo(String appId) {
            return () -> KeyBuilder.build("ADX:APP:APPID", appId);
        }

        /**
         * 应用信息
         * @param  platformType 平台类型：1 IOS，2 安卓，3 H5，4 PC
         */
        public static KeyGenerator getInfoByCustom(String customAppId, Integer platformType) {
            return () -> KeyBuilder.build("ADX:APP:CUSTOMAPPID", customAppId, "type", platformType);
        }

        public static KeyGenerator getAllPositionIdFromApp(String appId) {
            return () -> KeyBuilder.build("ADX:APP:ALL:POSITION", appId);
        }
    }

    /**
     * 广告位信息
     */
    public static class Position {

        /**
         * 广告位信息
         * TODO com.bxm.adx.common.sell.position.PositionDaoImpl 这里也有定义
         */
        public static KeyGenerator getInfo(String positionId) {
            return () -> KeyBuilder.build("ADX:POSITION:PID", positionId);
        }

        /**
         * 广告位信息
         */
        public static KeyGenerator getInfoByApp(String appPositionId) {
            return () -> KeyBuilder.build("ADX:POSITION:APPID", appPositionId);
        }

        public static KeyGenerator getPositionInteract() {
            return () -> KeyBuilder.build("ADX", "POSITION" ,"INTERACT");
        }
    }
    
    /**
     * 任务信息
     */
    public static class Task {

        /** 全部的任务列表 **/
        private final static String ALL = "ADX:SDK:TASK:ALL";
        /** 开启中的任务列表 **/
        private final static String OPENED = "ADX:SDK:TASK:OPENED";

        /**
         * 全部的任务列表
         */
        public static KeyGenerator getAllList() {
            return () -> ALL;
        }

        /**
         * 开启中的任务列表
         */
        public static KeyGenerator getOpenedList() {
            return () -> OPENED;
        }

        /**
         * 某任务绑定的区域列表
         * @param taskId 任务ID
         */
        public static KeyGenerator getRegionList(Long taskId) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "REGION");
        }

        /**
         * 某任务的开发者应用黑白名单规则
         * @param taskId 任务ID
         */
        public static KeyGenerator getAppRule(Long taskId) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "APP");
        }

        /**
         * 某任务的广告位黑白名单规则
         * @param taskId 任务ID
         */
        public static KeyGenerator getPositionRule(Long taskId) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "POSITION");
        }

        /**
         * 某任务的频次限制规则
         * @param taskId 任务ID
         */
        public static KeyGenerator getFrequencyRule(Long taskId) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "FREQ");
        }

        /**
         * 某任务的设备品牌名限制规则
         * @param taskId 任务ID
         */
        public static KeyGenerator getDeviceBrandRule(Long taskId) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "DEVBRAND");
        }

        /**
         * 某任务的小时段限制规则
         * @param taskId 任务ID
         */
        public static KeyGenerator getTimeSlotRule(Long taskId) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "TIMESLOT");
        }

        /**
         * 某任务的统计信息
         * @param taskId 任务ID
         */
        public static KeyGenerator getCounter(Long taskId) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId);
        }

        /**
         * 某任务某日的统计信息
         * @param taskId 任务ID
         * @param ymd 年月日
         */
        public static KeyGenerator getCounter(Long taskId, String ymd) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "date", ymd);
        }

        /**
         * 某任务某日某设备的统计信息
         * @param taskId 任务ID
         * @param ymd 年月日
         * @param devNo 设备号（imei或idfa）
         */
        public static KeyGenerator getCounter(Long taskId, String ymd, String devNo) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", taskId, "date", ymd, "dev", devNo);
        }

        /**
         * 某日某设备的统计信息
         * @param ymd 年月日
         * @param devNo 设备号（imei或idfa）
         */
        public static KeyGenerator getCounterForDev(String ymd, String devNo) {
            return () -> KeyBuilder.build("ADX:SDK:TASK", "date", ymd, "dev", devNo);
        }
    }

    /**
     * 统计信息
     */
    public static class Counter {

        /**
         * 某广告位某日某设备的第三方SDK统计信息(hash)
         * @param positionId 广告位ID
         * @param ymd 年月日
         * @param devNo 设备号（imei或idfa）
         */
        public static KeyGenerator getSdk(String positionId, String ymd, String devNo) {
            return () -> KeyBuilder.build("ADX:COUNT:SDK:PID", positionId, "date", ymd, "dev", devNo);
        }

        /**
         * 某任务某设备的最近下发时间戳
         * @param taskId 任务ID
         * @param devNo 设备号（imei或idfa）
         */
        public static KeyGenerator getTaskAcquireTime(Long taskId, String devNo) {
            return () -> KeyBuilder.build("ADX:COUNT:SDK:TASK:ACQ:TIMES", taskId, devNo);
        }
    }
}
