package com.bxm.adx.common.cipher;

import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.UUIDHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;

/**
 * @author allen
 * @date 2020-09-25
 * @since 1.0
 */
@Slf4j
@Service
public class CipherServiceImpl implements CipherService {

    private final Updater updater;
    private final Fetcher fetcher;

    public CipherServiceImpl(Updater updater, Fetcher fetcher) {
        this.updater = updater;
        this.fetcher = fetcher;
    }

    @Override
    public String aesEncrypt(String content, String key) {
        try {
            return encrypt(content, key);
        } catch (Exception e) {
            log.warn("encrypt: ", e);
            return null;
        }
    }

    @Override
    public String aesDecrypt(String encrypt, String key) {
        try {
            return decrypt(encrypt, key);
        } catch (Exception e) {
            log.warn("decrypt: ", e);
            return null;
        }
    }

    @Override
    public String generateRequestId(String key) {
        String requestId = UUIDHelper.generate();
        updater.update(getRequestId2Key(requestId), key, 10 * 60);
        return requestId;
    }

    @Override
    public String getKey(String requestId) {
        return fetcher.fetch(getRequestId2Key(requestId), String.class);
    }

    private static KeyGenerator getRequestId2Key(String requestId) {
        return () -> KeyBuilder.build("ADX", "TMP", "REQUESTID2KEY", requestId);
    }


    private static final String KEY_ALGORITHM  = "AES";

    private static String encrypt(String content, String key) throws Exception {
        if (StringUtils.isBlank(content) || StringUtils.isBlank(key)) {
            return null;
        }
        Cipher cipher = Cipher.getInstance(KEY_ALGORITHM);
        byte[] byteContent = content.getBytes(StandardCharsets.UTF_8);
        SecretKeySpec sKey = new SecretKeySpec(key.getBytes(), KEY_ALGORITHM);
        cipher.init(Cipher.ENCRYPT_MODE, sKey);
        byte[] result = cipher.doFinal(byteContent);
        return Base64.encodeBase64String(result);
    }

    private static String decrypt(String encryptContent, String key) throws Exception {
        byte[] content = Base64.decodeBase64(encryptContent);
        if (content == null) {
            return null;
        }
        SecretKeySpec sKey = new SecretKeySpec(key.getBytes(), KEY_ALGORITHM);
        Cipher cipher = Cipher.getInstance(KEY_ALGORITHM);
        cipher.init(Cipher.DECRYPT_MODE, sKey);
        byte[] result = cipher.doFinal(content);
        return new String(result);
    }
}
