package com.bxm.adx.common.filter;

import com.bxm.adx.common.ip.IpService;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.List;

/**
 * <h3>广告位地域限制</h3>
 *
 * @author hcmony
 * @since V1.0.0, 2020/11/23 10:46
 */
@Slf4j
@Component
public class PositionRegionLimiterFilter extends AbstractPositionFilter {
    /**
     * 无效区域后缀
     */
    private static final String ILLEGAL_REGION_CODE_SUFFIX="0000";

    private final IpService ipService;

    public PositionRegionLimiterFilter(IpService ipService) {
        this.ipService = ipService;
    }

    @Override
    protected List<String> doFilter(MarketOrders marketOrders) {
        final List<Position> positions = marketOrders.getPositions();
        if (CollectionUtils.isEmpty(positions)) {
            return null;
        }

        List<String> list = Lists.newArrayList();

        positions.forEach(position -> {
            String regionCodes = position.getRegion();
            if (StringUtils.isEmpty(regionCodes)) {
                return;
            }

            BidRequest bidRequest = marketOrders.getBidRequest();
            if (bidRequest != null) {
                Device device = bidRequest.getDevice();
                if (device != null) {
                    String ip = device.getIp();
                    if (!StringUtils.isEmpty(ip)) {
                        IP ipp = ipService.analyze(ip);
                        if (ipp != null) {
                            String regionCode = ipp.getRegioncode();
                            if (isOpen(regionCodes, regionCode)) {
                                return;
                            }
                        }
                    }
                }
            }

            list.add(position.getPositionId());
        });
        return list;
    }

    @Override
    public int order() {
        return 3;
    }

    private boolean isOpen(String regionCodes, String regionCode) {
        List<String> open = Arrays.asList(regionCodes.split(","));
        if (CollectionUtils.isEmpty(open)) {
            // 不限
            return true;
        }
        // 当行政编码为 xx0000，那么说明这个IP识别有误。如果可以投放的地区里面包含这个开头的，则投放。
        if (regionCode != null && regionCode.endsWith(ILLEGAL_REGION_CODE_SUFFIX)){
            for(String openRegin : open){
                if(getCodeForProvince(openRegin).equals(regionCode)){
                    return  true;
                };
            }
        }
        // 开放这个省、市、区县
        return open.contains(getCodeForProvince(regionCode)) || open.contains(getCodeForCity(regionCode)) || open.contains(regionCode);
    }

    private String getCodeForProvince(String regionCode) {
        return org.apache.commons.lang.StringUtils.rightPad(org.apache.commons.lang.StringUtils.left(regionCode, 2), 6, "0");
    }

    private String getCodeForCity(String regionCode) {
        return org.apache.commons.lang.StringUtils.rightPad(org.apache.commons.lang.StringUtils.left(regionCode, 4), 6, "0");
    }

}
