package com.bxm.adx.common.market.exchange;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;

import com.bxm.adx.common.adapter.AdxContext;
import com.bxm.adx.common.adapter.AdxContextFactory;
import org.apache.commons.lang.ArrayUtils;

import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.ingetration.AdxCounterServiceIntegration;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.request.Geo;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adxcounter.facade.constant.AdxMtEnum;
import com.bxm.adxcounter.facade.model.AdxCounterDTO;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;

/**
 * @author allen
 * @date 2020-10-30
 * @since 1.0
 */
@Slf4j
public class ExchangeCallable implements Callable<Deal> {

    private final Buyer buyer;
    private final BidRequest bidRequest;
    private final AdxCounterServiceIntegration service;
    private final BuyerMeter buyerMeter;
    private final ExchangeParam exchangeParam;

    public ExchangeCallable(Buyer buyer, BidRequest bidRequest, AdxCounterServiceIntegration service,
                            BuyerMeter buyerMeter, ExchangeParam exchangeParam) {
        this.buyer = buyer;
        this.bidRequest = bidRequest;
        this.service = service;
        this.buyerMeter = buyerMeter;
        this.exchangeParam = exchangeParam;
    }

    private void dot(BidRequest request, Buyer buyer, Integer mt) {
        List<AdxCounterDTO> dtos = build(request, buyer, mt);
        for (AdxCounterDTO dto : dtos) {
            try {
                service.counter(dto);
            } catch (Exception e) {
                log.warn("counter: ", e);
            }
        }
    }

    private List<AdxCounterDTO> build(BidRequest request, Buyer buyer, Integer mt) {
        List<AdxCounterDTO> dtos = Lists.newArrayList();
        List<Impression> imps = request.getImps();
        for (Impression imp : imps) {
            Device device = request.getDevice();

            AdxCounterDTO dto = new AdxCounterDTO();
            dto.setT(Objects.toString(System.currentTimeMillis()));
            dto.setIp(device.getIp());
            dto.setUa(device.getUa());
            dto.setMac(device.getMac());
            dto.setOs(device.isAndroid() ? AdxCounterDTO.OS_ANDROID : device.isIos() ? AdxCounterDTO.OS_IOS : AdxCounterDTO.OS_UNKNOWN);
            dto.setImei(device.getImei());
            dto.setImei_md5(device.getImei_md5());
            dto.setAnid(device.getDpid());
            dto.setAnid_md5(device.getDpid_md5());
            dto.setOaid(device.getOaid());
            dto.setIdfa(device.getIdfa());
            dto.setIdfa_md5(device.getIdfa_md5());
            dto.setDevb(null);
            dto.setDevm(null);
            dto.setNet(getNet(device));

            Geo geo = device.getGeo();
            if (Objects.nonNull(geo)) {
                dto.setLon(Objects.toString(geo.getLon()));
                dto.setLat(Objects.toString(geo.getLat()));
            }
            dto.setMt(mt);
            dto.setBidid(request.getId());
            dto.setTagid(imp.getTag_id());
            Dsp dsp = buyer.getDsp();
            if (Objects.isNull(dsp)) {
                continue;
            }
            dto.setDspid(Objects.toString(dsp.getId()));
            dto.setCreateid(null);
            dto.setWin(null);
            dto.setStatus(null);
            dto.setActid(null);
            dto.setScene(null);

            dtos.add(dto);
        }
        return dtos;
    }

    private String getNet(Device device) {
        Integer connectionType = device.getConnection_type();
        return Objects.toString(connectionType);
    }

    @Override
    public Deal call() {
        String name = buyer.getCode();
        BuyModelAdapter modelAdapter = buyer.getModelAdapter();
        if (null == modelAdapter) {
            log.warn("BuyModelAdapter [{}] not found!", name);
            return null;
        }

        try {
            dot(bidRequest, buyer, AdxMtEnum._102.getOriginal());

            // 对上下文设置必要的对象
            ExchangeContext.putBidRequest(bidRequest);
            ExchangeContext.putBuyer(buyer);
            ExchangeContext.putDispatchConfig(exchangeParam.getDispatcherConfigId());

            byte[] request = modelAdapter.buildRequest(bidRequest);
            if (ArrayUtils.isEmpty(request)) {
                return null;
            }

            // 请求指标
            buyerMeter.increaseRequest(buyer);

            byte[] response = offer(request);
            if (ArrayUtils.isEmpty(response)) {
                return null;
            }

            Deal deal = new Deal(buyer, request, response, bidRequest);

            if (!deal.isBidSuccess()) {
                return null;
            }

            // 填充指标
            buyerMeter.increasePadding(buyer);

            dot(bidRequest, buyer, AdxMtEnum._103.getOriginal());

            return deal;
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("deal call err", e);
            }
            return null;
        } finally {
            ExchangeContext.remove();
        }
    }

    private byte[] offer(byte[] request) {
        long start = System.nanoTime();
        byte[] response;
        try {
            response = buyer.offer(request);
        } finally {
            // 耗时指标
            buyerMeter.recordRequest(buyer, start);
        }
        return response;
    }
}
