package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.adapter.AdxContext;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.ingetration.AdxCounterServiceIntegration;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.*;

/**
 * @author allen
 * @since 2019-12-12
 */
@Slf4j
@Configuration
public class RtbExchanger implements Exchanger {

    private final AdxProperties properties;
    private final AdxCounterServiceIntegration service;
    private final BuyerMeter buyerMeter;
    private final Counter counter;
    public RtbExchanger(AdxProperties properties, AdxCounterServiceIntegration service, BuyerMeter buyerMeter, Counter counter) {
        this.properties = properties;
        this.service = service;
        this.buyerMeter = buyerMeter;
        this.counter = counter;
    }

    @Override
    public List<Deal> bidding(BidRequest bidRequest, Collection<Buyer> buyers) {
        int size = buyers.size();
        if (0 == size) {
            return Collections.emptyList();
        }
        if (null == bidRequest) {
            return Collections.emptyList();
        }

        ThreadPoolExecutor executor = new ThreadPoolExecutor(size, size, 5, TimeUnit.SECONDS, new LinkedBlockingQueue<>(), new NamedThreadFactory("exchange"));
        executor.allowCoreThreadTimeOut(true);

        ExchangeParam exchangeParam = new ExchangeParam();
        exchangeParam.setDispatcherConfigId(AdxContextFactory.get().getConfigId());

        List<Future<Deal>> futures = Lists.newArrayListWithCapacity(size);
        for (final Buyer buyer : buyers) {
            if (!dspQPSLimit(buyer.getDsp())) {
                continue;
            }
            Future<Deal> future = executor.submit(new ExchangeCallable(buyer, bidRequest, service, buyerMeter, exchangeParam));
            futures.add(future);
        }

        List<Deal> deals = Lists.newArrayList();
        for (Future<Deal> future : futures) {
            try {
                Deal response = future.get();
//                Deal response = future.get(properties.getBuyerBiddingTimeOutInMillis(), TimeUnit.MILLISECONDS);
                if (null != response) {
                    deals.add(response);
                }
            } catch (InterruptedException | ExecutionException e) {
                // 超时或意外终止
                future.cancel(true);
                log.error("execute: ", e);
            }
        }
        return deals;
    }

    private boolean dspQPSLimit(Dsp dsp) {
        if (dsp == null) {
            return false;
        }
        Integer qps = dsp.getQps();
        if (qps == null) {
            return true;
        }
        if (qps.intValue() < 0) {
            return true;
        }
        if (qps.intValue() == 0) {
            return false;
        }
        long time = LocalDateTime.now().toEpochSecond(ZoneOffset.of("+8"));
        KeyGenerator keyGenerator = CacheKeys.getDspNowQPS(time+"");
        Long now = counter.hincrementAndGet(keyGenerator, dsp.getDspCode(), 600);
        if (now > qps) {
//            counter.hincrementAndGet(CacheKeys.getDspQPSBan(time+""), dsp.getDspCode(), 600);
            return false;
        } else {
            return true;
        }
    }
}
