package com.bxm.adx.common.buy.dispatcher;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;

import com.bxm.adx.common.CacheKeys;
import com.bxm.mccms.facade.model.pushable.DispatcherDspCacheVO;
import org.apache.commons.collections.MapUtils;
import org.springframework.stereotype.Component;

import com.bxm.adx.common.buy.dsp.DspService;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionService;
import com.bxm.adx.facade.constant.pushable.CachePushableFields;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;

/**
 * For cache pushable
 *
 * <pre>
 *     name = "DISPATCHER"
 *     parameters = {positionId=$positionId}
 *     byte[] = Dispatcher object to json bytes
 * </pre>
 *
 * @author allen
 * @since 2019-12-19
 */
@CachePush("DISPATCHER")
@Component
public class DispatcherDaoImpl implements DispatcherDao, Pushable {

    private final Fetcher fetcher;
    private final Updater updater;
    private final PositionService positionService;
    private final DspService dspService;
    private final DispatcherAB dispatcherAB;

    public DispatcherDaoImpl(Fetcher fetcher, Updater updater, PositionService positionService, DspService dspService,
                             DispatcherAB dispatcherAB) {
        this.fetcher = fetcher;
        this.updater = updater;
        this.positionService = positionService;
        this.dspService = dspService;
        this.dispatcherAB = dispatcherAB;
    }

    @Override
    public Collection<Dispatcher> get(String positionId) {
        Map<String, Dispatcher> map = fetcher.hfetchall(CacheKeys.Dispather.getKeyGenerator(positionId), Dispatcher.class);
        if (MapUtils.isEmpty(map)) {
            map = fetcher.hfetchall(getKeyGenerator(positionId), Dispatcher.class);
        }
        return MapUtils.isNotEmpty(map) ? map.values() : null;
    }

    @Override
    public Dispatcher getByAppPosId(String appPosId, String dspCode) {
        Dispatcher dispatcher = fetcher.hfetch(CacheKeys.Dispather.getKeyGeneratorByAppPosId(appPosId), dspCode, Dispatcher.class);
        return Objects.isNull(dispatcher) ?
                fetcher.hfetch(getKeyGeneratorByAppPosId(appPosId), dspCode, Dispatcher.class) : dispatcher;
    }

    @Override
    public Dispatcher get(String positionId, String dspId) {
        Dispatcher dispatcher = fetcher.hfetch(CacheKeys.Dispather.getKeyGenerator(positionId), dspId, Dispatcher.class);
        return Objects.isNull(dispatcher) ?
                fetcher.hfetch(getKeyGenerator(positionId), dspId, Dispatcher.class) : dispatcher;
    }

    @Override
    public Collection<DispatcherDspCacheVO> getDispatchersByConfigId(String positionId, Long configId) {
        return dispatcherAB.getDispatchersByConfigId(positionId, configId);
    }

    @Override
    public Dispatcher getByConfig(String positionId, String dspId, Long configId) {
        return fetcher.hfetch(CacheKeys.Dispather.getKeyGeneratorByConfig(positionId), configId + "-" + dspId, Dispatcher.class);
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        Object o = parameters.get(CachePushableFields.POSITION_ID);
        if (Objects.isNull(o)) {
            return;
        }
        String positionId = Objects.toString(o);
        Dispatcher dispatcher = JsonHelper.convert(data, Dispatcher.class);
        String dspId = Objects.toString(dispatcher.getDspId());
        String dspCode = dspService.get(Long.valueOf(dspId)).getDspCode();// FIXME: 2020/2/21 dspcode要保证唯一性
        updater.hremove(getKeyGenerator(positionId), dspId);
        updater.hupdate(getKeyGenerator(positionId), dspId, dispatcher);

        Position position = positionService.getByPositionId(positionId);
        updater.hremove(getKeyGeneratorByAppPosId(position.getAppPositionId()), dspCode);
        updater.hupdate(getKeyGeneratorByAppPosId(position.getAppPositionId()), dspCode, dispatcher);
    }

    private KeyGenerator getKeyGenerator(String positionId) {
        return () -> KeyBuilder.build("ADX", "DISPATHER", "PID", positionId);
    }

    private KeyGenerator getKeyGeneratorByAppPosId(String appPosId) {
        return () -> KeyBuilder.build("ADX", "DISPATHER", "APPID", appPosId);
    }
}
