package com.bxm.adx.common.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.utils.AdxUtils;
import com.bxm.mcssp.common.entity.PositionJsonConfig;
import com.bxm.mcssp.common.enums.position.PositionSceneTypeEnum;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Objects;

/**
 * <h3>推送广告位间隔限制过滤器</h3>
 * 曝光
 *
 * @author hcmony
 * @since V1.0.0, 2020/11/23 10:46
 */
@Slf4j
@Component
public class PositionPushIntervalTimesFilter extends AbstractPositionFilter {
    @Autowired
    private Counter counter;

    @Override
    protected List<String> doFilter(MarketOrders marketOrders) {
        if (marketOrders.getBidRequest().isAppDrivingRequest()) {
            return null;
        }

        final List<Position> positions = marketOrders.getPositions();
        if (CollectionUtils.isEmpty(positions)) {
            return null;
        }

        List<String> list = Lists.newArrayList();

        //需要注意的是这里的position信息已经全部传过来了
        positions.forEach(position -> {
            Integer intervalTime = getIntervalTime(position);//分钟
            if (null == intervalTime || intervalTime <= 0) {
                return;
            }

            //数据来自于adxcounter
            KeyGenerator key = () -> KeyBuilder.build("statistics", "adx", "show", "user", "lasttime", position.getPositionId(), marketOrders.getUid());
            final Long lastTime = counter.get(key);


            if (lastTime == null) {
                return;
            }

            final long now = System.currentTimeMillis();

            if  (now > (lastTime + (intervalTime * 60 * 1000) - (10 * 1000))) {
                return;
            }

            list.add(position.getPositionId());
        });

        return list;
    }

    /**
     * 获取广告位曝光间隔
     *
     * @param position
     * @return
     */
    private Integer getIntervalTime(Position position) {
        Byte pScene = position.getPositionScene();
        if (Objects.nonNull(pScene)) {
            PositionSceneTypeEnum typeEnum = PositionSceneTypeEnum.get(pScene.intValue());
            if (Objects.nonNull(typeEnum)) {
                String jsonConfig = position.getJsonConfig();
                if (!StringUtils.isEmpty(jsonConfig)) {
                    PositionJsonConfig positionJsonConfig = AdxUtils.getJsonConfig(jsonConfig);
                    if (Objects.nonNull(positionJsonConfig)) {
                        switch (typeEnum) {
                            case PUSH:   //推送广告位
                                PositionJsonConfig.Push push = positionJsonConfig.getPush();
                                if (push == null) {
                                    return null;
                                }
                                if (Objects.nonNull(push.getMinIntervalBetweenTwoExposures())) {
                                    return push.getMinIntervalBetweenTwoExposures();
                                } else {
                                    if (Objects.nonNull(position.getPushIntervalTime())) {
                                        return position.getPushIntervalTime();
                                    } else {
                                        return null;
                                    }
                                }
                            case SCREEN: //插屏
                                PositionJsonConfig.Screen screen = positionJsonConfig.getScreen();
                                if (screen == null) {
                                    return null;
                                }
                                return screen.getMinIntervalBetweenTwoExposures();
                            default:
                                return null;
                        }
                    }
                }
            }
        }
        return null;
    }

    @Override
    public int order() {
        return 2;
    }
}
