package com.bxm.adx.common.market.exchange;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.cache.BuyerResponseCache;
import com.bxm.adx.common.buy.position.AdvertPoint;
import com.bxm.adx.common.buy.position.AdvertPointService;
import com.bxm.adx.common.log.datalog.DataLogDao;
import com.bxm.adx.common.sell.BidConfig;
import com.bxm.adx.common.sell.request.*;
import com.bxm.warcar.utils.JsonHelper;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;

import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.ingetration.AdxCounterServiceIntegration;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.openlog.event.internal.AdxRequestEvent;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.adxcounter.facade.constant.AdxMtEnum;
import com.bxm.adxcounter.facade.model.AdxCounterDTO;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.google.common.collect.Lists;

import lombok.extern.slf4j.Slf4j;

/**
 * @author allen
 * @date 2020-10-30
 * @since 1.0
 */
@Slf4j
public class ExchangeCallable implements Callable<Deal> {

    private final Buyer buyer;
    private final BidRequest bidRequest;
    private final AdxCounterServiceIntegration service;
    private final BuyerMeter buyerMeter;
    private final ExchangeParam exchangeParam;
    private final EventPark eventPark;
    private final BuyerResponseCache buyerResponseCache;
    private final CountDownLatch waitCountDown;
    private final CountDownLatch overtimeCountDown;
    private final AdvertPointService advertPointService;
    private final BidConfig bidConfig;
    private final DataLogDao dataLogDao;

    public ExchangeCallable(Buyer buyer, BidRequest bidRequest, AdxCounterServiceIntegration service,
                            BuyerMeter buyerMeter, ExchangeParam exchangeParam, EventPark eventPark,
                            BuyerResponseCache buyerResponseCache,
                            CountDownLatch waitCountDown, CountDownLatch overtimeCountDown,
                            AdvertPointService advertPointService, BidConfig bidConfig, DataLogDao dataLogDao) {

        this.buyer = buyer;
        this.bidRequest = bidRequest;
        this.service = service;
        this.buyerMeter = buyerMeter;
        this.exchangeParam = exchangeParam;
        this.eventPark = eventPark;
        this.buyerResponseCache = buyerResponseCache;
        this.waitCountDown = waitCountDown;
        this.overtimeCountDown = overtimeCountDown;
        this.advertPointService = advertPointService;
        this.bidConfig = bidConfig;
        this.dataLogDao = dataLogDao;
    }

    private void dot(List<AdxCounterDTO> dtos, Integer mt) {
        for (AdxCounterDTO dto : dtos) {
            try {
                dto.setMt(mt);
                service.counter(dto);
            } catch (Exception e) {
                log.warn("counter: ", e);
            }
        }
    }

    private List<AdxCounterDTO> build(BidRequest request, Buyer buyer) {
        List<AdxCounterDTO> dtos = Lists.newArrayList();
        List<Impression> imps = request.getImps();
        for (Impression imp : imps) {
            Device device = request.getDevice();

            AdxCounterDTO dto = new AdxCounterDTO();
            dto.setT(Objects.toString(System.currentTimeMillis()));
            dto.setIp(device.getIp());
            dto.setUa(device.getUa());
            dto.setMac(device.getMac());
            dto.setOs(device.isAndroid() ? AdxCounterDTO.OS_ANDROID : device.isIos() ? AdxCounterDTO.OS_IOS : AdxCounterDTO.OS_UNKNOWN);
            dto.setImei(device.getImei());
            dto.setImei_md5(device.getImei_md5());
            dto.setAnid(device.getDpid());
            dto.setAnid_md5(device.getDpid_md5());
            dto.setOaid(device.getOaid());
            dto.setIdfa(device.getIdfa());
            dto.setIdfa_md5(device.getIdfa_md5());
            dto.setDevb(null);
            dto.setDevm(null);
            dto.setNet(getNet(device));

            Geo geo = device.getGeo();
            if (Objects.nonNull(geo)) {
                dto.setLon(Objects.toString(geo.getLon()));
                dto.setLat(Objects.toString(geo.getLat()));
            }
            dto.setBidid(request.getId());
            dto.setTagid(imp.getTag_id());
            Dsp dsp = buyer.getDsp();
            if (Objects.isNull(dsp)) {
                continue;
            }
            dto.setDspid(Objects.toString(dsp.getId()));
            dto.setCreateid(null);
            dto.setWin(null);
            dto.setStatus(null);
            dto.setActid(null);
            dto.setScene(null);

            dtos.add(dto);
        }
        return dtos;
    }

    private String getNet(Device device) {
        Integer connectionType = device.getConnection_type();
        return Objects.toString(connectionType);
    }

    @Override
    public Deal call() {
        long startTime = System.currentTimeMillis();

        String name = buyer.getCode();
        BuyModelAdapter modelAdapter = buyer.getModelAdapter();
        if (null == modelAdapter) {
            log.warn("BuyModelAdapter [{}] not found!", name);
            return null;
        }

        List<AdxCounterDTO> counterDTOS = build(bidRequest, buyer);
        byte[] request = null;

        BidRequest adxRequest = null;
        Deal deal = null;
        try {
//            dot(counterDTOS, AdxMtEnum._102.getOriginal());

            // 对上下文设置必要的对象
            ExchangeContext.putBidRequest(bidRequest);
            ExchangeContext.putBuyer(buyer);
            ExchangeContext.putDispatchConfig(exchangeParam.getDispatcherConfigId());
            Dispatcher dispatcher = exchangeParam.getDispatcher();
            ExchangeContext.putDispatch(dispatcher);

            adxRequest = rebuildBidRequest(exchangeParam);
            ExchangeContext.putAdxRequest(adxRequest);
            request = modelAdapter.buildRequest(adxRequest);
            if (ArrayUtils.isEmpty(request)) {
                return null;
            }
            Dsp dsp = buyer.getDsp();
            String configId = StringUtils.EMPTY;
            if (Objects.nonNull(dispatcher) && Objects.nonNull(dispatcher.getConfigId())) {
                configId = dispatcher.getConfigId().toString();
            }
            //埋点adx请求
            eventPark.post(new AdxRequestEvent(this, bidRequest, adxRequest, dsp.getId().toString(), configId,
                    bidConfig));

            // 请求指标
            buyerMeter.increaseRequest(buyer);

            byte[] response = offer(request);
            if (ArrayUtils.isEmpty(response)) {
                throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
            }

            deal = new Deal(buyer, request, response, bidRequest, adxRequest);

            if (!deal.isBidSuccess()) {
                throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
            }

            rebuildResponse(deal, exchangeParam, dsp);
            //埋点dsp返回
            eventPark.post(new DspBidEvent(this, bidRequest, adxRequest, deal.getBidResponse(),
                    dsp.getId().toString(), bidConfig));

            // 填充指标
            buyerMeter.increasePadding(buyer);

//            dot(counterDTOS, AdxMtEnum._103.getOriginal());

            return deal;
        } catch (AdxException e) {
            AdxErrEnum adxErrEnum = Optional.ofNullable(e.getAdxErrEnum()).orElse(AdxErrEnum.UNKNOWN_ERR);
            return new Deal(buyer, request, bidRequest, adxRequest, adxErrEnum);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("deal call err", e);
            }
            return new Deal(buyer, request, bidRequest, adxRequest, AdxErrEnum.UNKNOWN_ERR);
        } finally {
            long endTime = System.currentTimeMillis();
            long interval = endTime - startTime;
            if (log.isDebugEnabled()) {
                log.debug("buy = {}, interval = {}", JsonHelper.convert(buyer), interval);
            }
            if (interval < bidRequest.getWaitTime()) {
                waitCountDown.countDown();
            } else if (interval < bidRequest.getOvertime()) {
                overtimeCountDown.countDown();
            } else {
                if (Objects.nonNull(deal) && deal.isBidSuccess()) {
                    buyerResponseCache.saveResponse(deal.getAdxRequest(), deal.getBidResponse(), buyer);
                }
            }
            ExchangeContext.remove();
        }
    }

    private byte[] offer(byte[] request) {
        long start = System.nanoTime();
        byte[] response;
        try {
            response = buyer.offer(request);
        } finally {
            // 耗时指标
            buyerMeter.recordRequest(buyer, start);
        }
        return response;
    }

    /**
     * 重构bidrequest
     * 1.判断广告位接入类型SDK/API决定是否使用流量分配中的广告位id
     * 2.流量方/流量分配两者的底价选择
     * 3.流量方/流量分配两者的APPID选择
     *
     * @param exchangeParam
     * @return
     */
    private BidRequest rebuildBidRequest(ExchangeParam exchangeParam) {
        BidRequest adxRequest = new BidRequest();
        BeanUtils.copyProperties(bidRequest, adxRequest);
        Dispatcher dispatcher = exchangeParam.getDispatcher();
        List<Impression> impressions = adxRequest.getImps();
        Integer dockingMethodType = adxRequest.getDockingMethodType();
        List<Impression> newImpressions = new ArrayList<>();
        AdvertPoint advertPoint = null;
        for (Impression impression : impressions) {
            if (Objects.isNull(impression.getBid_floor())) {
                impression.setBid_floor(BigDecimal.ZERO);
            }
            Impression newImpression = new Impression();
            BeanUtils.copyProperties(impression, newImpression);
            //替换DSP广告位ID
            if (Objects.nonNull(dockingMethodType) &&
                    (dockingMethodType == DockingMethodTypeEnum.SDK_OPERATION.getType())) {
                if (buyer.getCode().equalsIgnoreCase("scene")) {
                    //场景dsp直接用SSP入口广告位
                } else {
                    newImpression.setTag_id(dispatcher.getDspPosid());
                }
            } else {
                if (Objects.isNull(advertPoint)) {
                    advertPoint = advertPointService.getAdvertPoint(dispatcher.getDspAppid(), dispatcher.getDspPosid());
                }
                if (Objects.nonNull(advertPoint) && StringUtils.isNotEmpty(advertPoint.getTemplateId())) {
                    Native an = newImpression.getA_native();
                    if (Objects.isNull(an)) {
                        an = new Native();
                    }
                    an.setTemplate(advertPoint.getTemplateId());
                    newImpression.setA_native(an);
                }
                ExchangeContext.putDataLog(dataLogDao.existDataLog(impression.getTag_id(), dispatcher.getDspPosid()));
                newImpression.setTag_id(dispatcher.getDspPosid());
            }
            //优先使用流量分配中的底价
            BigDecimal dspBasePrice = dispatcher.getDspBasePrice();
            if (Objects.nonNull(dspBasePrice) && dspBasePrice.movePointRight(2).compareTo(impression.getBid_floor()) > 0) {
                newImpression.setBid_floor(dispatcher.getDspBasePrice().movePointRight(2));
            }
            if (Objects.isNull(newImpression.getBid_floor())) {
                newImpression.setBid_floor(BigDecimal.ZERO);
            }
            //如果利润率不为空，则计算向流量方的出价替换原有DSP的广告出价 （新出价 = （100 - 利润率）* DSP出价）
            if (Objects.nonNull(dispatcher.getProfitMargin())) {
                BigDecimal floor = newImpression.getBid_floor().divide(new BigDecimal(100).subtract(dispatcher.getProfitMargin()).movePointLeft(2), 2, BigDecimal.ROUND_UP);
                newImpression.setBid_floor(floor);
            }

            //底价单位为分，不保留小数，有小数自动向上进位
            newImpression.setBid_floor(newImpression.getBid_floor().setScale(0, BigDecimal.ROUND_UP));
            newImpressions.add(newImpression);
            break;
        }
        adxRequest.setImps(newImpressions);

        App app = new App();
        if (Objects.nonNull(adxRequest.getApp())) {
            BeanUtils.copyProperties(adxRequest.getApp(), app);
        }
        app.setId(dispatcher.getDspAppid());
        app.setBundle(dispatcher.getAppPackageName());
        if (Objects.nonNull(advertPoint)) {
            app.setVer(advertPoint.getVersion());
            app.setName(advertPoint.getAppName());
        }
        
        adxRequest.setApp(app);
        return adxRequest;
    }

    /**
     * 重构Response
     *
     * @param deal
     * @param exchangeParam
     */
    private void rebuildResponse(Deal deal, ExchangeParam exchangeParam, Dsp dsp) {
        if (deal.isOp()) {
            return;
        }

        BidResponse bidResponse = deal.getBidResponse();
        BidRequest bidRequest = deal.getBidRequest();
        Dispatcher dispatcher = exchangeParam.getDispatcher();
        if (StringUtils.isEmpty(bidResponse.getId())) {
            bidResponse.setId(bidRequest.getId());
        }

        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            seatBid.setDspId(dispatcher.getDspId());
            seatBid.setConfigId(dispatcher.getConfigId());
            for (Bid bid : seatBid.getBid()) {
                //记录dsp出价
                bid.setDsp_price(new BigDecimal(bid.getPrice().toString()));
                if (Objects.nonNull(dispatcher.getDspAvgPrice())) {
                    bid.setPrice(dispatcher.getDspAvgPrice().movePointRight(2));
                } else {
                    bid.setPrice(bid.getPrice());
                }
                //如果利润率不为空，则计算向流量方的出价替换原有DSP的广告出价 （新出价 = （100 - 利润率）* DSP出价）
                if (Objects.nonNull(dispatcher.getProfitMargin())) {
                    BigDecimal price = bid.getPrice();
                    price = price.multiply(new BigDecimal(100).subtract(dispatcher.getProfitMargin())).movePointLeft(2);
                    bid.setPrice(price);
                }
                //出价舍去小数（向流量方出价更低，提高收益）
                bid.setPrice(bid.getPrice().setScale(0, BigDecimal.ROUND_DOWN));
                bid.setDsp_appid(exchangeParam.getDispatcher().getDspAppid());
            }
        }
    }

    /**
     * 判断dsp合作类型是否是实时竞价
     *
     * @param dsp
     * @return
     */
    private boolean isRtb(Dsp dsp) {
        if (Objects.nonNull(dsp)) {
            Integer settleType = dsp.getSettleType();
            if (Objects.nonNull(settleType) && settleType == AdxConstants.SettleType.REAL_TIME_RTB.getType()) {
                return true;
            }
        }
        return false;
    }
}
