package com.bxm.adx.common.sell.position;

import java.util.Map;
import java.util.Objects;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.mccms.facade.model.pushable.PositionInteractAppentranceRefCacheVO;
import com.bxm.warcar.xcache.TargetFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import com.bxm.adx.facade.constant.pushable.CachePushableFields;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import org.springframework.util.StringUtils;

/**
 * For cache pushable
 *
 * <pre>
 * name = "POSITION"
 * parameters = {positionId=$positionId}
 * byte[] = Position object to json bytes
 * </pre>
 *
 * @author allen
 * @since 2019-12-18
 */
@CachePush("POSITION")
@Component
@Slf4j
public class PositionDaoImpl implements PositionDao, Pushable {

    private final Fetcher fetcher;
    private final Updater updater;
    private final com.bxm.warcar.xcache.Fetcher fetcher2;
    private final AdxProperties properties;
    private static final String APPID = "APPID";
    private static final String PID = "PID";
    private static final String TYPE = "TYPE";

    public PositionDaoImpl(Fetcher fetcher, Updater updater, com.bxm.warcar.xcache.Fetcher fetcher2, AdxProperties properties) {
        this.fetcher = fetcher;
        this.updater = updater;
        this.fetcher2 = fetcher2;
        this.properties = properties;
    }

    @Override
    public Position getByAppPositionId(String appPositionId) {
        return fetcher.fetch(getKeyGenerator(APPID, appPositionId), Position.class);
    }

    @Override
    public Position getByPositionId(String positionId) {
        return fetcher.fetch(getKeyGenerator(PID, positionId), Position.class);
    }

    @Override
    public PositionInteractAppentranceRefCacheVO getInteractByPositionId(String positionId) {
        PositionInteractAppentranceRefCacheVO cacheVO = fetcher.hfetch(AdxKeyGenerator.Position.getPositionInteract(), positionId, PositionInteractAppentranceRefCacheVO.class);
        return cacheVO;
    }

    @Override
    public Position getByPositionIdNativeCache(String positionId) {

        return fetcher2.fetch(new TargetFactory<Position>()
                .cls(Position.class)
                .keyGenerator(getKeyGenerator(PID, positionId))
                .skipNativeCache(false)
                .expireTimeInSecond(600)
                .build());
    }

    @Override
    public Position getByAppPositionIdNativeCache(String appPositionId) {
        return fetcher2.fetch(new TargetFactory<Position>()
                .cls(Position.class)
                .keyGenerator(getKeyGenerator(APPID, appPositionId))
                .skipNativeCache(false)
                .expireTimeInSecond(600)
                .build());
    }

    @Override
    public Position getByPositionId(String appId, Integer w, Integer h, Integer positionType, String appPositionId) {
        return fetcher2.fetch(new TargetFactory<Position>()
                .cls(Position.class)
                .keyGenerator(getKeyGenerator(TYPE, buildKeyVal(appId, w, h, positionType, appPositionId)))
                .skipNativeCache(false)
                .expireTimeInSecond(600)
                .build());
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        if (log.isWarnEnabled()) {
            log.warn("position data = {}", JsonHelper.convert(data, String.class));
        }
        Object o = parameters.get(CachePushableFields.POSITION_ID);
        if (Objects.isNull(o)) {
            return;
        }
        String positionId = Objects.toString(o);
        Position position = JsonHelper.convert(data, Position.class);

        updater.remove(getKeyGenerator(APPID, position.getAppPositionId()));
        updater.remove(getKeyGenerator(PID, positionId));
        if (!StringUtils.isEmpty(position.getPositionSize()) && position.getAppId().equals(properties.getBesAppId())) {
            updater.remove(getKeyGenerator(TYPE, buildKeyVal(properties.getBesAppId(), position)));
        }
        String appId = position.getAppId();
        if (!StringUtils.isEmpty(appId)) {
            updater.srem(AdxKeyGenerator.App.getAllPositionIdFromApp(appId), positionId);
        }
        if (position.isEnabled()) {
            updater.update(getKeyGenerator(APPID, position.getAppPositionId()), position);
            updater.update(getKeyGenerator(PID, positionId), position);
            if (!StringUtils.isEmpty(position.getPositionSize()) && position.getAppId().equals(properties.getBesAppId())) {
                updater.update(getKeyGenerator(TYPE, buildKeyVal(properties.getBesAppId(), position)), position);
            }
            if (!StringUtils.isEmpty(appId)) {
                updater.supdate(AdxKeyGenerator.App.getAllPositionIdFromApp(appId), positionId);
            }
        }
    }

    // TODO com.bxm.adx.facade.constant.redis.AdxKeyGenerator.Position 这里也有定义
    private KeyGenerator getKeyGenerator(String name, String value) {
        return () -> KeyBuilder.build("ADX", "POSITION", name, value);
    }

    private String buildKeyVal(String appId, Position position) {
        String positionSize = position.getPositionSize().split(" ")[0];
        String appPositionId = position.getAppPositionId();
        String val = new StringBuilder(appId)
                .append("-")
                .append(appPositionId)
                .append("-")
                .append(position.getPositionScene())
                .append("-")
                .append(positionSize)
                .toString();
        return val;
    }

    private String buildKeyVal(String appId, Integer w, Integer h, Integer positionType, String appPositionId) {
        String val = new StringBuilder(appId)
                .append("-")
                .append(appPositionId)
                .append("-")
                .append(positionType)
                .append("-")
                .append(w)
                .append("*")
                .append(h)
                .toString();
        return val;
    }
}
