package com.bxm.adx.common.log.datalog;

import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.xcache.TargetFactory;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author fgf
 * @date 2022/7/26
 **/
@CachePush("DATA_LOG")
@Component
@Slf4j
public class DataLogDaoImpl implements DataLogDao, Pushable {
    private final Map<String, AtomicInteger> cache = Maps.newConcurrentMap();
    private final Updater updater;

    public DataLogDaoImpl(Updater updater) {
        this.updater = updater;
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        if (log.isWarnEnabled()) {
            log.warn("datalog data = {}", JsonHelper.convert(data, String.class));
        }

        DataLog dataLog = JsonHelper.convert(data, DataLog.class);
        if (StringUtils.isEmpty((dataLog.getSspPositionId())) || StringUtils.isEmpty((dataLog.getAdvertPointPositionId()))) {
            return;
        }
        KeyGenerator keyGenerator = getKeyGenerator(dataLog.getSspPositionId(), dataLog.getAdvertPointPositionId());
        updater.remove(keyGenerator);
        updater.update(keyGenerator, dataLog, 600);
        cache.put(mapKey(dataLog.getSspPositionId(), dataLog.getAdvertPointPositionId()), new AtomicInteger(500));
    }

    private KeyGenerator getKeyGenerator(String sspPositionId, String dspPositionId) {
        return () -> KeyBuilder.build("ADX", "DATA_LOG", sspPositionId, dspPositionId);
    }

    @Override
    public boolean existDataLog(String sspPositionId, String dspPositionId) {
        String key = mapKey(sspPositionId, dspPositionId);
        boolean exist = cache.containsKey(key);
        if (exist) {
            int i = cache.get(key).getAndDecrement();
            if (i <= 0) {
                cache.remove(key);
            }
        }
        return exist;
    }

    private String mapKey(String sspPositionId, String dspPositionId) {
        return sspPositionId + "-" + dspPositionId;
    }
}
